# Panduan Sistem Migrasi Database

Sistem migrasi database memungkinkan Anda mengelola update schema secara otomatis tanpa perlu update manual setiap kali ada perubahan.

## 🚀 Quick Start

### 1. Menjalankan Migrasi yang Pending

**Via Web Browser:**
1. Login sebagai Admin atau Manager
2. Klik menu "🔧 Migrations" di navbar
3. Klik tombol "Run Pending Migrations"

**Via Command Line:**
```bash
php migrate.php run
```

### 2. Cek Status Migrasi

**Via Web:**
- Akses menu "Migrations" di navbar

**Via CLI:**
```bash
php migrate.php status
```

### 3. Rollback (Jika Perlu)

**Via Web:**
- Klik "Rollback Last Batch" di halaman migrations

**Via CLI:**
```bash
php migrate.php rollback
```

## 📝 Membuat Migration Baru

### Cara 1: Menggunakan Helper Script (Recommended)

```bash
php create_migration.php nama_migration
```

Contoh:
```bash
php create_migration.php add_discount_field
```

Ini akan membuat file: `migrations/20240115120000_add_discount_field.php`

### Cara 2: Manual

1. Copy template:
```bash
cp migrations/.template.php migrations/YYYYMMDDHHMMSS_nama_migration.php
```

2. Edit file dan implementasikan `up()` dan `down()` methods

## 📋 Contoh Migration

```php
<?php
/**
 * Migration: Add Discount Field
 * Date: 2024-01-15
 */

class AddDiscountField extends Migration {
    
    public function up() {
        // Check jika column belum ada
        if (!$this->columnExists('transactions', 'discount')) {
            // Tambah column
            $this->execute("ALTER TABLE transactions 
                ADD COLUMN discount DECIMAL(10,2) DEFAULT 0 AFTER total");
        }
    }
    
    public function down() {
        // Rollback: hapus column jika ada
        if ($this->columnExists('transactions', 'discount')) {
            $this->execute("ALTER TABLE transactions DROP COLUMN discount");
        }
    }
}
```

## 🛠️ Helper Methods

Migration class menyediakan helper methods:

### `columnExists($table, $column)`
Check jika column sudah ada di table.

```php
if (!$this->columnExists('users', 'email')) {
    $this->execute("ALTER TABLE users ADD COLUMN email VARCHAR(255)");
}
```

### `tableExists($table)`
Check jika table sudah ada.

```php
if (!$this->tableExists('new_table')) {
    $this->execute("CREATE TABLE new_table (...)");
}
```

### `execute($sql)`
Execute SQL query dengan error handling otomatis.

```php
$this->execute("UPDATE users SET status = 'active'");
```

## 📁 Struktur File

```
pos/
├── migrations/
│   ├── README.md                          # Dokumentasi detail
│   ├── .template.php                      # Template migration
│   ├── 20240101000001_add_transaction_items_status.php
│   ├── 20240101000002_add_waiters_dapur_roles.php
│   ├── 20240101000003_add_payment_status.php
│   └── 20240101000004_add_transaction_tables_merge.php
├── app/
│   └── classes/
│       └── Migrator.php                    # Class migrator
├── migrate.php                            # CLI & Web interface
└── create_migration.php                   # Helper untuk buat migration baru
```

## ✅ Best Practices

1. **Selalu Check Sebelum Execute**
   - Gunakan `columnExists()` atau `tableExists()` sebelum ALTER/CREATE
   - Migration harus idempotent (bisa dijalankan berkali-kali tanpa error)

2. **Implementasikan Rollback**
   - Selalu buat method `down()` yang benar
   - Test rollback sebelum deploy

3. **Naming Convention**
   - Format: `YYYYMMDDHHMMSS_descriptive_name.php`
   - Class name: PascalCase dari descriptive_name
   - Contoh: `20240115120000_add_discount.php` → Class: `AddDiscount`

4. **Test Migration**
   - Test di development environment dulu
   - Pastikan tidak ada data loss
   - Test rollback juga

## 🔍 Troubleshooting

### Migration gagal
- Check error message di console/web
- Pastikan SQL syntax benar
- Pastikan tidak ada conflict dengan existing schema
- Check database connection

### Migration sudah dijalankan tapi masih pending
- Check table `migrations` di database
- Pastikan nama migration file sesuai dengan yang di database

### Rollback gagal
- Pastikan `down()` method sudah diimplementasikan dengan benar
- Check jika perubahan sudah bisa di-rollback (misal: data sudah ada)

## 📊 Migration Tracking

Sistem menggunakan table `migrations` untuk tracking:
- `migration_name` - Nama file migration
- `batch` - Batch number (untuk rollback)
- `executed_at` - Waktu execution
- `execution_time` - Durasi execution

## 🎯 Workflow

1. **Development:**
   ```bash
   # Buat migration baru
   php create_migration.php add_new_feature
   
   # Edit migration file
   # Implement up() dan down()
   
   # Test migration
   php migrate.php run
   php migrate.php rollback
   php migrate.php run
   ```

2. **Production:**
   ```bash
   # Cek status
   php migrate.php status
   
   # Run pending migrations
   php migrate.php run
   ```

## 📚 Migration Files yang Sudah Ada

1. **20240101000001_add_transaction_items_status.php**
   - Menambah status column ke transaction_items
   - Menambah completed_at column

2. **20240101000002_add_waiters_dapur_roles.php**
   - Menambah role waiters dan dapur
   - Membuat user waiters dan dapur default

3. **20240101000003_add_payment_status.php**
   - Menambah payment_status dan paid_at ke transactions
   - Support fitur "Bayar Nanti"

4. **20240101000004_add_transaction_tables_merge.php**
   - Menambah table transaction_tables
   - Support merge multiple tables

## 💡 Tips

- Selalu backup database sebelum run migration di production
- Test migration di staging environment dulu
- Dokumentasikan perubahan di migration file
- Gunakan transaction untuk migration yang kompleks (sudah otomatis di Migrator)






