# Arsitektur Web POS Offline-First

## 📋 Ringkasan

Sistem Web POS telah diubah menjadi **offline-first** menggunakan Progressive Web App (PWA) dengan Service Worker dan IndexedDB. Sistem dapat berjalan penuh saat internet mati dan melakukan sinkronisasi otomatis saat koneksi kembali.

## 🏗️ Arsitektur Sistem

```
┌─────────────────────────────────────────────────────────────┐
│                    Browser (Client)                          │
│  ┌──────────────────────────────────────────────────────┐   │
│  │              Service Worker (sw.js)                   │   │
│  │  • Cache static assets (HTML, CSS, JS)               │   │
│  │  • Network-first strategy dengan cache fallback      │   │
│  │  • Background sync untuk transaksi                  │   │
│  └──────────────────────────────────────────────────────┘   │
│                                                              │
│  ┌──────────────────────────────────────────────────────┐   │
│  │         IndexedDB (offline-db.js)                   │   │
│  │  • transactions (transaksi offline)                   │   │
│  │  • transaction_items (detail item)                   │   │
│  │  • products_snapshot (snapshot produk)               │   │
│  │  • categories_snapshot (snapshot kategori)            │   │
│  │  • sync_queue (antrian sinkronisasi)                 │   │
│  └──────────────────────────────────────────────────────┘   │
│                                                              │
│  ┌──────────────────────────────────────────────────────┐   │
│  │         Sync Engine (sync-engine.js)                 │   │
│  │  • Deteksi online/offline                            │   │
│  │  • Sinkronisasi otomatis saat online                │   │
│  │  • Retry mechanism dengan exponential backoff        │   │
│  │  • Idempotent API calls                              │   │
│  └──────────────────────────────────────────────────────┘   │
│                                                              │
│  ┌──────────────────────────────────────────────────────┐   │
│  │    Status Indicator (offline-status.js)              │   │
│  │  • Tampilkan status koneksi (Online/Offline)         │   │
│  │  • Jumlah transaksi pending                          │   │
│  │  • Status sinkronisasi                               │   │
│  └──────────────────────────────────────────────────────┘   │
└─────────────────────────────────────────────────────────────┘
                            ↕ HTTP/HTTPS
┌─────────────────────────────────────────────────────────────┐
│                    Server (PHP)                              │
│  ┌──────────────────────────────────────────────────────┐   │
│  │         API: sync-transaction.php                    │   │
│  │  • Menerima transaksi dari client                   │   │
│  │  • Validasi dan simpan ke database                  │   │
│  │  • Idempotent (cegah duplikasi)                     │   │
│  └──────────────────────────────────────────────────────┘   │
│                                                              │
│  ┌──────────────────────────────────────────────────────┐   │
│  │         Database (MySQL)                              │   │
│  │  • transactions                                       │   │
│  │  • transaction_items                                 │   │
│  │  • products                                          │   │
│  │  • categories                                        │   │
│  └──────────────────────────────────────────────────────┘   │
└─────────────────────────────────────────────────────────────┘
```

## 🔄 Flow Transaksi Offline → Online

### 1. **Saat Offline (Internet Mati)**

```
User Input Transaksi
    ↓
Validasi di Client (JavaScript)
    ↓
Simpan ke IndexedDB
    ├─ transactions (status: PENDING_SYNC)
    ├─ transaction_items
    └─ sync_queue (status: PENDING)
    ↓
Tampilkan Konfirmasi ke User
    ↓
Clear Cart & Reset Form
```

**Status Transaksi:**
- `sync_status`: `PENDING_SYNC`
- `created_at`: Timestamp lokal
- `invoice_number`: `OFFLINE-{timestamp}-{random}`

### 2. **Saat Online (Internet Kembali)**

```
Deteksi Online Event
    ↓
Sync Engine Triggered
    ↓
Ambil Pending Transactions dari IndexedDB
    ↓
Untuk setiap transaksi:
    ├─ POST ke /api/sync-transaction.php
    ├─ Server validasi & simpan ke DB
    ├─ Update sync_status = SYNCED
    └─ Update sync_queue status
    ↓
Jika semua berhasil:
    └─ Update UI: "Semua transaksi tersinkronisasi"
    
Jika ada yang gagal:
    └─ Update UI: "X transaksi gagal, akan dicoba lagi"
```

### 3. **Idempotent API**

Server mencegah duplikasi dengan:
- Check `invoice_number` sebelum insert
- Jika sudah ada, return existing transaction ID
- Client update `sync_status` tanpa duplikasi

## 💾 Struktur Database Lokal (IndexedDB)

### **Object Store: transactions**
```javascript
{
  id: autoIncrement,
  invoice_number: string (unique),
  local_id: string,
  items: array,
  total: number,
  payment_method: string,
  cash_received: number,
  table_ids: array | null,
  payment_status: 'paid' | 'unpaid',
  sync_status: 'PENDING_SYNC' | 'SYNCED' | 'FAILED',
  server_id: number | null,
  created_at: ISO string,
  user_id: number
}
```

**Indexes:**
- `sync_status` (untuk query pending)
- `created_at` (untuk sorting)
- `invoice_number` (unique)

### **Object Store: transaction_items**
```javascript
{
  id: autoIncrement,
  transaction_id: number,
  product_id: number,
  product_name: string,
  product_code: string,
  quantity: number,
  price: number,
  subtotal: number
}
```

**Indexes:**
- `transaction_id` (untuk get items per transaction)

### **Object Store: products_snapshot**
```javascript
{
  id: number,
  name: string,
  code: string,
  price: number,
  stock: number,
  category_id: number,
  // ... fields lainnya
}
```

**Indexes:**
- `code` (unique)
- `category_id`

### **Object Store: categories_snapshot**
```javascript
{
  id: number,
  name: string,
  // ... fields lainnya
}
```

### **Object Store: sync_queue**
```javascript
{
  id: autoIncrement,
  transaction_id: number,
  transaction_data: object,
  status: 'PENDING' | 'SYNCING' | 'SYNCED' | 'FAILED',
  created_at: ISO string,
  retry_count: number
}
```

**Indexes:**
- `transaction_id`
- `status`
- `created_at`

## 🔧 Komponen Utama

### 1. **Service Worker (sw.js)**

**Fungsi:**
- Cache static assets (HTML, CSS, JS)
- Network-first strategy dengan cache fallback
- Background sync untuk transaksi
- Offline page fallback

**Strategi Caching:**
- **Static Assets**: Cache saat install, serve dari cache jika offline
- **Dynamic Content**: Network first, fallback ke cache
- **API Calls**: Tidak di-cache, ditangani oleh sync engine

### 2. **OfflineDB (offline-db.js)**

**Class: `OfflineDB`**

**Methods:**
- `init()` - Initialize database
- `saveTransaction(data)` - Simpan transaksi offline
- `getPendingTransactions()` - Ambil transaksi pending
- `updateSyncStatus(id, status, serverId)` - Update status sync
- `saveProductsSnapshot(products)` - Simpan snapshot produk
- `getProductsSnapshot()` - Ambil snapshot produk
- `addToSyncQueue(transactionId, data)` - Tambah ke sync queue
- `getSyncQueue()` - Ambil sync queue

### 3. **Sync Engine (sync-engine.js)**

**Class: `SyncEngine`**

**Fungsi:**
- Deteksi online/offline
- Sinkronisasi otomatis saat online
- Retry mechanism (max 3 retries)
- Periodic sync (setiap 30 detik)
- Background sync registration

**Events:**
- `syncStatusUpdate` - Status sync berubah
- `syncComplete` - Sync selesai

### 4. **Status Indicator (offline-status.js)**

**Class: `OfflineStatusIndicator`**

**Fungsi:**
- Tampilkan status koneksi (Online/Offline)
- Jumlah transaksi pending
- Status sinkronisasi
- Tombol manual sync

**UI:**
- Status bar di top page
- Warna hijau (online) / orange (offline)
- Animasi pulse saat offline
- Info jumlah pending transactions

## 📱 PWA Manifest

**File: `manifest.json`**

**Features:**
- Installable app
- Standalone display mode
- Icons untuk berbagai ukuran
- Shortcuts untuk POS dan Dapur
- Theme color: #6366F1

## 🔐 Keamanan & Best Practices

### 1. **Idempotent API**
- Server check `invoice_number` sebelum insert
- Mencegah duplikasi transaksi
- Return existing ID jika sudah ada

### 2. **Data Validation**
- Validasi di client (JavaScript)
- Validasi di server (PHP)
- Double-check untuk integrity

### 3. **Conflict Resolution**
- **Strategy**: Server is source of truth
- Transaksi bersifat append-only (tidak bisa edit)
- Jika conflict, server version wins

### 4. **Error Handling**
- Retry mechanism dengan exponential backoff
- Max 3 retries per transaction
- Log errors untuk debugging
- User notification untuk errors

## ⚠️ Batasan Mode Offline

### **Yang BISA dilakukan offline:**
✅ Membuat transaksi baru
✅ Menghitung total, pajak, diskon
✅ Menambah/mengurangi item di cart
✅ Melihat produk dan kategori (dari snapshot)
✅ Menyimpan transaksi ke IndexedDB

### **Yang TIDAK BISA dilakukan offline:**
❌ Update stok real-time (menggunakan snapshot)
❌ Laporan real-time (menunggu sync)
❌ Edit transaksi yang sudah dibuat
❌ Hapus transaksi (append-only)
❌ Update produk/kategori (read-only dari snapshot)

### **Catatan Penting:**
- **Stok**: Menggunakan snapshot saat offline, mungkin tidak akurat
- **Laporan**: Menunggu sinkronisasi untuk data terbaru
- **Transaksi**: Tidak bisa di-edit setelah dibuat (append-only)

## 🚀 Cara Menggunakan

### **1. Install sebagai PWA:**
1. Buka aplikasi di browser
2. Klik "Install" atau "Add to Home Screen"
3. Aplikasi akan terinstall dan bisa dibuka offline

### **2. Operasional Offline:**
1. Buka aplikasi (akan load dari cache jika offline)
2. Buat transaksi seperti biasa
3. Transaksi akan tersimpan di IndexedDB
4. Status indicator akan menampilkan jumlah pending

### **3. Sinkronisasi:**
- **Otomatis**: Saat internet kembali, sync engine akan otomatis sync
- **Manual**: Klik tombol "Sync" di status bar
- **Background**: Service Worker akan trigger background sync

## 📊 Monitoring & Debugging

### **Console Logs:**
- `[SW]` - Service Worker logs
- `[OfflineDB]` - IndexedDB operations
- `[SyncEngine]` - Sync operations
- `[POS]` - POS-specific logs

### **Browser DevTools:**
- **Application > Service Workers**: Cek status SW
- **Application > IndexedDB**: Lihat data offline
- **Application > Cache Storage**: Lihat cached assets
- **Network**: Monitor API calls

## 🔄 Update & Maintenance

### **Update Service Worker:**
1. Update `sw.js` dengan version baru
2. Service Worker akan auto-update saat reload
3. Old cache akan dihapus otomatis

### **Update Database Schema:**
1. Increment version di `offline-db.js`
2. Handle migration di `onupgradeneeded`
3. Test dengan data existing

## 📝 Checklist Implementasi

- [x] PWA Manifest
- [x] Service Worker dengan caching
- [x] IndexedDB schema
- [x] Offline transaction flow
- [x] Sync engine
- [x] Status indicator
- [x] API endpoint untuk sync
- [x] Products & categories snapshot
- [x] Error handling & retry
- [x] Idempotent API

## 🎯 Next Steps (Optional)

1. **Push Notifications**: Notifikasi saat sync berhasil/gagal
2. **Offline Reports**: Generate laporan dari IndexedDB
3. **Conflict UI**: UI untuk resolve conflicts
4. **Export/Import**: Export transaksi offline untuk backup
5. **Multi-device Sync**: Sync antar device (jika diperlukan)

---

**Dokumen ini menjelaskan arsitektur lengkap sistem POS offline-first. Untuk pertanyaan atau update, silakan hubungi tim development.**

