<?php
/**
 * Authentication Class
 * Handles user authentication and authorization
 */

class Auth {
    private $db;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    /**
     * Login user
     */
    public function login($username, $password) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ? AND status = 'active'");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            
            if (verifyPassword($password, $user['password'])) {
                // Set session
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user'] = [
                    'id' => $user['id'],
                    'username' => $user['username'],
                    'name' => $user['name'],
                    'role' => $user['role']
                ];
                
                // Update last login
                $stmt = $this->db->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                $stmt->bind_param("i", $user['id']);
                $stmt->execute();
                
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Logout user
     */
    public function logout() {
        $_SESSION = array();
        
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time()-3600, '/');
        }
        
        session_destroy();
    }
    
    /**
     * Check if user has permission
     */
    public function hasPermission($requiredRole) {
        $user = getCurrentUser();
        if (!$user) {
            return false;
        }
        
        // Define role hierarchy (lower index = higher privilege)
        $roles = ['admin', 'manager', 'cashier', 'waiters', 'dapur'];
        $userRoleIndex = array_search($user['role'], $roles);
        $requiredRoleIndex = array_search($requiredRole, $roles);
        
        // For waiters and dapur, use specific permissions
        if ($user['role'] == 'waiters' || $user['role'] == 'dapur') {
            return ($user['role'] == $requiredRole);
        }
        
        return $userRoleIndex !== false && $userRoleIndex <= $requiredRoleIndex;
    }
    
    /**
     * Require permission
     */
    public function requirePermission($requiredRole) {
        requireLogin();
        
        if (!$this->hasPermission($requiredRole)) {
            setFlash('error', 'Anda tidak memiliki akses untuk halaman ini.');
            redirect(BASE_URL . 'public/dashboard.php');
        }
    }
}
?>

