<?php
/**
 * Database Migrator Class
 * Handles database migrations automatically
 */

class Migrator {
    private $db;
    private $migrationsPath;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->migrationsPath = __DIR__ . '/../../migrations';
        
        // Create migrations table if not exists
        $this->createMigrationsTable();
    }
    
    /**
     * Create migrations table to track applied migrations
     */
    private function createMigrationsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS migrations (
            id INT AUTO_INCREMENT PRIMARY KEY,
            migration_name VARCHAR(255) UNIQUE NOT NULL,
            batch INT NOT NULL,
            executed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            execution_time DECIMAL(10,3) DEFAULT 0
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $this->db->query($sql);
    }
    
    /**
     * Get all migration files from migrations directory
     */
    public function getMigrationFiles() {
        $files = [];
        if (is_dir($this->migrationsPath)) {
            $files = glob($this->migrationsPath . '/*.php');
            // Sort by filename (which should include timestamp)
            sort($files);
        }
        return $files;
    }
    
    /**
     * Get applied migrations from database
     */
    public function getAppliedMigrations() {
        $stmt = $this->db->prepare("SELECT migration_name FROM migrations ORDER BY id");
        $stmt->execute();
        $result = $stmt->get_result();
        $migrations = [];
        while ($row = $result->fetch_assoc()) {
            $migrations[] = $row['migration_name'];
        }
        return $migrations;
    }
    
    /**
     * Get pending migrations
     */
    public function getPendingMigrations() {
        $allFiles = $this->getMigrationFiles();
        $applied = $this->getAppliedMigrations();
        $pending = [];
        
        foreach ($allFiles as $file) {
            $migrationName = basename($file, '.php');
            if (!in_array($migrationName, $applied)) {
                $pending[] = $file;
            }
        }
        
        return $pending;
    }
    
    /**
     * Get migration class name from filename
     */
    private function getMigrationClassName($filePath) {
        $filename = basename($filePath, '.php');
        // Convert filename like 20240101000000_add_payment_status.php to AddPaymentStatus
        $parts = explode('_', $filename);
        // Remove timestamp (first part if it's numeric)
        if (is_numeric($parts[0]) && strlen($parts[0]) >= 8) {
            array_shift($parts);
        }
        // Convert to PascalCase
        $className = '';
        foreach ($parts as $part) {
            $className .= ucfirst($part);
        }
        return $className;
    }
    
    /**
     * Run a single migration
     */
    public function runMigration($filePath) {
        $migrationName = basename($filePath, '.php');
        $className = $this->getMigrationClassName($filePath);
        
        // Check if already applied
        $applied = $this->getAppliedMigrations();
        if (in_array($migrationName, $applied)) {
            return ['success' => false, 'message' => "Migration {$migrationName} already applied"];
        }
        
        // Include migration file
        require_once $filePath;
        
        if (!class_exists($className)) {
            return ['success' => false, 'message' => "Class {$className} not found in migration file"];
        }
        
        $migration = new $className($this->db);
        
        if (!($migration instanceof Migration)) {
            return ['success' => false, 'message' => "Migration class must extend Migration base class"];
        }
        
        $startTime = microtime(true);
        
        try {
            $this->db->autocommit(false);
            
            // Run up migration
            $migration->up();
            
            // Get current batch number
            $batchStmt = $this->db->prepare("SELECT MAX(batch) as max_batch FROM migrations");
            $batchStmt->execute();
            $batchResult = $batchStmt->get_result();
            $batch = ($batchResult->fetch_assoc()['max_batch'] ?? 0) + 1;
            
            // Record migration
            $executionTime = microtime(true) - $startTime;
            $recordStmt = $this->db->prepare("INSERT INTO migrations (migration_name, batch, execution_time) VALUES (?, ?, ?)");
            $recordStmt->bind_param("sid", $migrationName, $batch, $executionTime);
            $recordStmt->execute();
            
            $this->db->commit();
            
            return [
                'success' => true, 
                'message' => "Migration {$migrationName} applied successfully",
                'execution_time' => round($executionTime, 3)
            ];
            
        } catch (Exception $e) {
            $this->db->rollback();
            return [
                'success' => false, 
                'message' => "Error running migration {$migrationName}: " . $e->getMessage()
            ];
        } finally {
            $this->db->autocommit(true);
        }
    }
    
    /**
     * Run all pending migrations
     */
    public function runPendingMigrations() {
        $pending = $this->getPendingMigrations();
        $results = [];
        
        if (empty($pending)) {
            return ['success' => true, 'message' => 'No pending migrations', 'results' => []];
        }
        
        foreach ($pending as $file) {
            $result = $this->runMigration($file);
            $results[] = [
                'file' => basename($file),
                'result' => $result
            ];
        }
        
        $allSuccess = true;
        foreach ($results as $r) {
            if (!$r['result']['success']) {
                $allSuccess = false;
                break;
            }
        }
        
        return [
            'success' => $allSuccess,
            'message' => count($pending) . ' migration(s) processed',
            'results' => $results
        ];
    }
    
    /**
     * Rollback last batch of migrations
     */
    public function rollbackLastBatch() {
        // Get last batch
        $stmt = $this->db->prepare("SELECT MAX(batch) as max_batch FROM migrations");
        $stmt->execute();
        $result = $stmt->get_result();
        $maxBatch = $result->fetch_assoc()['max_batch'] ?? 0;
        
        if ($maxBatch == 0) {
            return ['success' => false, 'message' => 'No migrations to rollback'];
        }
        
        // Get migrations in last batch
        $stmt = $this->db->prepare("SELECT migration_name FROM migrations WHERE batch = ? ORDER BY id DESC");
        $stmt->bind_param("i", $maxBatch);
        $stmt->execute();
        $result = $stmt->get_result();
        $migrations = [];
        while ($row = $result->fetch_assoc()) {
            $migrations[] = $row['migration_name'];
        }
        
        $results = [];
        
        foreach ($migrations as $migrationName) {
            $filePath = $this->migrationsPath . '/' . $migrationName . '.php';
            if (!file_exists($filePath)) {
                $results[] = [
                    'migration' => $migrationName,
                    'success' => false,
                    'message' => 'Migration file not found'
                ];
                continue;
            }
            
            $className = $this->getMigrationClassName($filePath);
            require_once $filePath;
            
            if (!class_exists($className)) {
                $results[] = [
                    'migration' => $migrationName,
                    'success' => false,
                    'message' => "Class {$className} not found"
                ];
                continue;
            }
            
            $migration = new $className($this->db);
            
            try {
                $this->db->autocommit(false);
                $migration->down();
                
                // Remove from migrations table
                $deleteStmt = $this->db->prepare("DELETE FROM migrations WHERE migration_name = ?");
                $deleteStmt->bind_param("s", $migrationName);
                $deleteStmt->execute();
                
                $this->db->commit();
                
                $results[] = [
                    'migration' => $migrationName,
                    'success' => true,
                    'message' => 'Rolled back successfully'
                ];
            } catch (Exception $e) {
                $this->db->rollback();
                $results[] = [
                    'migration' => $migrationName,
                    'success' => false,
                    'message' => $e->getMessage()
                ];
            } finally {
                $this->db->autocommit(true);
            }
        }
        
        return [
            'success' => true,
            'message' => 'Rollback completed',
            'results' => $results
        ];
    }
    
    /**
     * Get migration status
     */
    public function getStatus() {
        $allFiles = $this->getMigrationFiles();
        $applied = $this->getAppliedMigrations();
        $pending = $this->getPendingMigrations();
        
        return [
            'total' => count($allFiles),
            'applied' => count($applied),
            'pending' => count($pending),
            'pending_files' => array_map('basename', $pending)
        ];
    }
}

/**
 * Base Migration Class
 * All migrations must extend this class
 */
abstract class Migration {
    protected $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    /**
     * Run migration (forward)
     */
    abstract public function up();
    
    /**
     * Rollback migration (backward)
     */
    abstract public function down();
    
    /**
     * Helper: Check if column exists
     */
    protected function columnExists($table, $column) {
        // Escape table name (can't use prepared statement for table name)
        $table = $this->db->real_escape_string($table);
        $column = $this->db->real_escape_string($column);
        $result = $this->db->query("SHOW COLUMNS FROM `{$table}` LIKE '{$column}'");
        return $result && $result->num_rows > 0;
    }
    
    /**
     * Helper: Check if table exists
     */
    protected function tableExists($table) {
        $table = $this->db->real_escape_string($table);
        $dbName = $this->db->real_escape_string($this->db->query("SELECT DATABASE()")->fetch_row()[0]);
        $result = $this->db->query("SELECT COUNT(*) as count 
            FROM information_schema.tables 
            WHERE table_schema = '{$dbName}' 
            AND table_name = '{$table}'");
        if ($result) {
            $row = $result->fetch_assoc();
            return $row['count'] > 0;
        }
        return false;
    }
    
    /**
     * Helper: Execute SQL safely
     */
    protected function execute($sql) {
        if (!$this->db->query($sql)) {
            throw new Exception("SQL Error: " . $this->db->error . " | SQL: " . substr($sql, 0, 200));
        }
    }
}
?>

