<?php
/**
 * Product Class
 * Handles product operations
 */

class Product {
    private $db;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    /**
     * Get all products
     */
    public function getAll($category_id = null) {
        if ($category_id) {
            $stmt = $this->db->prepare("SELECT p.*, c.name as category_name 
                FROM products p 
                LEFT JOIN categories c ON p.category_id = c.id 
                WHERE p.category_id = ? AND p.status = 'active'
                ORDER BY p.name");
            $stmt->bind_param("i", $category_id);
        } else {
            $stmt = $this->db->prepare("SELECT p.*, c.name as category_name 
                FROM products p 
                LEFT JOIN categories c ON p.category_id = c.id 
                WHERE p.status = 'active'
                ORDER BY p.name");
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $products = [];
        while ($row = $result->fetch_assoc()) {
            $products[] = $row;
        }
        return $products;
    }
    
    /**
     * Get product by ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT p.*, c.name as category_name 
            FROM products p 
            LEFT JOIN categories c ON p.category_id = c.id 
            WHERE p.id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }
    
    /**
     * Create product
     */
    public function create($data) {
        $stmt = $this->db->prepare("INSERT INTO products (name, code, category_id, price, stock, description, status) 
            VALUES (?, ?, ?, ?, ?, ?, 'active')");
        $stmt->bind_param("ssiids", 
            $data['name'], 
            $data['code'], 
            $data['category_id'], 
            $data['price'], 
            $data['stock'], 
            $data['description']
        );
        return $stmt->execute();
    }
    
    /**
     * Update product
     */
    public function update($id, $data) {
        $stmt = $this->db->prepare("UPDATE products SET 
            name = ?, code = ?, category_id = ?, price = ?, stock = ?, description = ? 
            WHERE id = ?");
        $stmt->bind_param("ssiidsi", 
            $data['name'], 
            $data['code'], 
            $data['category_id'], 
            $data['price'], 
            $data['stock'], 
            $data['description'],
            $id
        );
        return $stmt->execute();
    }
    
    /**
     * Delete product (soft delete)
     */
    public function delete($id) {
        $stmt = $this->db->prepare("UPDATE products SET status = 'inactive' WHERE id = ?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }
    
    /**
     * Update stock
     */
    public function updateStock($id, $quantity) {
        $stmt = $this->db->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
        $stmt->bind_param("ii", $quantity, $id);
        return $stmt->execute();
    }
}
?>






