<?php
/**
 * Table Class
 * Handles table management operations
 */

class Table {
    private $db;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    /**
     * Get all tables
     */
    public function getAll() {
        $stmt = $this->db->prepare("SELECT t.*, 
            COUNT(tr.id) as active_transactions,
            MAX(tr.created_at) as last_transaction
            FROM tables t
            LEFT JOIN transactions tr ON t.id = tr.table_id 
                AND tr.status = 'completed' 
                AND DATE(tr.created_at) = CURDATE()
            GROUP BY t.id
            ORDER BY t.name");
        $stmt->execute();
        $result = $stmt->get_result();
        $tables = [];
        while ($row = $result->fetch_assoc()) {
            $tables[] = $row;
        }
        return $tables;
    }
    
    /**
     * Get table by ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM tables WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }
    
    /**
     * Get available tables
     */
    public function getAvailable() {
        $stmt = $this->db->prepare("SELECT * FROM tables WHERE status = 'available' ORDER BY name");
        $stmt->execute();
        $result = $stmt->get_result();
        $tables = [];
        while ($row = $result->fetch_assoc()) {
            $tables[] = $row;
        }
        return $tables;
    }
    
    /**
     * Get occupied tables
     */
    public function getOccupied() {
        $stmt = $this->db->prepare("SELECT t.*, 
            tr.id as transaction_id,
            tr.invoice_number,
            tr.created_at as order_time,
            COUNT(ti.id) as total_items,
            SUM(CASE WHEN ti.status = 'ready' THEN 1 ELSE 0 END) as ready_items
            FROM tables t
            INNER JOIN transactions tr ON t.id = tr.table_id
            LEFT JOIN transaction_items ti ON tr.id = ti.transaction_id
            WHERE t.status = 'occupied' 
                AND tr.status = 'completed'
                AND DATE(tr.created_at) = CURDATE()
            GROUP BY t.id, tr.id
            ORDER BY tr.created_at DESC");
        $stmt->execute();
        $result = $stmt->get_result();
        $tables = [];
        while ($row = $result->fetch_assoc()) {
            $tables[] = $row;
        }
        return $tables;
    }
    
    /**
     * Get tables with ready orders (for waiters)
     */
    public function getTablesWithReadyOrders() {
        $stmt = $this->db->prepare("SELECT DISTINCT t.*, 
            tr.id as transaction_id,
            tr.invoice_number,
            tr.created_at as order_time,
            COUNT(DISTINCT CASE WHEN ti.status = 'ready' THEN ti.id END) as ready_items,
            COUNT(DISTINCT ti.id) as total_items
            FROM tables t
            INNER JOIN transactions tr ON t.id = tr.table_id
            INNER JOIN transaction_items ti ON tr.id = ti.transaction_id
            WHERE ti.status = 'ready'
                AND tr.status = 'completed'
                AND DATE(tr.created_at) = CURDATE()
            GROUP BY t.id, tr.id
            ORDER BY tr.created_at ASC");
        $stmt->execute();
        $result = $stmt->get_result();
        $tables = [];
        while ($row = $result->fetch_assoc()) {
            $tables[] = $row;
        }
        return $tables;
    }
    
    /**
     * Create table
     */
    public function create($name, $capacity = 4, $location = '', $notes = '') {
        $stmt = $this->db->prepare("INSERT INTO tables (name, capacity, location, notes, status) VALUES (?, ?, ?, ?, 'available')");
        $stmt->bind_param("siss", $name, $capacity, $location, $notes);
        return $stmt->execute();
    }
    
    /**
     * Update table
     */
    public function update($id, $name, $capacity = 4, $location = '', $notes = '') {
        $stmt = $this->db->prepare("UPDATE tables SET name = ?, capacity = ?, location = ?, notes = ? WHERE id = ?");
        $stmt->bind_param("sissi", $name, $capacity, $location, $notes, $id);
        return $stmt->execute();
    }
    
    /**
     * Update table status
     */
    public function updateStatus($id, $status) {
        $valid_statuses = ['available', 'occupied', 'reserved', 'cleaning'];
        if (!in_array($status, $valid_statuses)) {
            return false;
        }
        
        $stmt = $this->db->prepare("UPDATE tables SET status = ? WHERE id = ?");
        $stmt->bind_param("si", $status, $id);
        return $stmt->execute();
    }
    
    /**
     * Delete table (soft delete by setting status)
     */
    public function delete($id) {
        // Check if table has active transactions
        $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM transactions 
            WHERE table_id = ? AND DATE(created_at) = CURDATE() AND status != 'cancelled'");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['count'] > 0) {
            return false; // Cannot delete table with active transactions
        }
        
        // Set status to cleaning (soft delete)
        return $this->updateStatus($id, 'cleaning');
    }
    
    /**
     * Get table status with current order info
     * Only returns active order if there are items that are not all completed
     */
    public function getTableStatus($table_id) {
        // Try to get from transaction_tables first (new system)
        $stmt = $this->db->prepare("SELECT t.*, 
            tr.id as current_transaction_id,
            tr.invoice_number,
            tr.total,
            tr.created_at as order_time,
            COUNT(ti.id) as total_items,
            SUM(CASE WHEN ti.status = 'pending' THEN 1 ELSE 0 END) as pending_items,
            SUM(CASE WHEN ti.status = 'preparing' THEN 1 ELSE 0 END) as preparing_items,
            SUM(CASE WHEN ti.status = 'ready' THEN 1 ELSE 0 END) as ready_items,
            SUM(CASE WHEN ti.status = 'completed' THEN 1 ELSE 0 END) as completed_items
            FROM tables t
            LEFT JOIN transaction_tables tt ON t.id = tt.table_id
            LEFT JOIN transactions tr ON tt.transaction_id = tr.id
                AND tr.status = 'completed' 
                AND DATE(tr.created_at) = CURDATE()
            LEFT JOIN transaction_items ti ON tr.id = ti.transaction_id
            WHERE t.id = ?
            GROUP BY t.id, tr.id
            HAVING current_transaction_id IS NOT NULL 
                AND COALESCE(total_items, 0) > 0
                AND COALESCE(completed_items, 0) < COALESCE(total_items, 0)
            ORDER BY tr.created_at DESC
            LIMIT 1");
        $stmt->bind_param("i", $table_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $status = $result->fetch_assoc();
        
        // If not found via transaction_tables, try old method (backward compatibility)
        if (!$status || !$status['current_transaction_id']) {
            $stmt2 = $this->db->prepare("SELECT t.*, 
                tr.id as current_transaction_id,
                tr.invoice_number,
                tr.total,
                tr.created_at as order_time,
                COUNT(ti.id) as total_items,
                SUM(CASE WHEN ti.status = 'pending' THEN 1 ELSE 0 END) as pending_items,
                SUM(CASE WHEN ti.status = 'preparing' THEN 1 ELSE 0 END) as preparing_items,
                SUM(CASE WHEN ti.status = 'ready' THEN 1 ELSE 0 END) as ready_items,
                SUM(CASE WHEN ti.status = 'completed' THEN 1 ELSE 0 END) as completed_items
                FROM tables t
                LEFT JOIN transactions tr ON t.id = tr.table_id
                    AND tr.status = 'completed' 
                    AND DATE(tr.created_at) = CURDATE()
                LEFT JOIN transaction_items ti ON tr.id = ti.transaction_id
                WHERE t.id = ?
                GROUP BY t.id, tr.id
                HAVING current_transaction_id IS NOT NULL 
                    AND COALESCE(total_items, 0) > 0
                    AND COALESCE(completed_items, 0) < COALESCE(total_items, 0)
                ORDER BY tr.created_at DESC
                LIMIT 1");
            $stmt2->bind_param("i", $table_id);
            $stmt2->execute();
            $result2 = $stmt2->get_result();
            $status2 = $result2->fetch_assoc();
            if ($status2 && $status2['current_transaction_id']) {
                return $status2;
            }
        }
        
        return $status;
    }
    
    /**
     * Get tables that can be merged (occupied tables with active transactions today)
     */
    public function getMergeableTables($exclude_table_id = null) {
        $sql = "SELECT DISTINCT t.*, 
            tr.id as transaction_id,
            tr.invoice_number,
            tr.created_at as order_time
            FROM tables t
            INNER JOIN transaction_tables tt ON t.id = tt.table_id
            INNER JOIN transactions tr ON tt.transaction_id = tr.id
            WHERE t.status = 'occupied' 
                AND tr.status = 'completed'
                AND DATE(tr.created_at) = CURDATE()";
        
        if ($exclude_table_id) {
            $sql .= " AND t.id != ?";
        }
        
        $sql .= " ORDER BY t.name";
        
        if ($exclude_table_id) {
            $stmt = $this->db->prepare($sql);
            $stmt->bind_param("i", $exclude_table_id);
        } else {
            $stmt = $this->db->prepare($sql);
        }
        
        $stmt->execute();
        $result = $stmt->get_result();
        $tables = [];
        while ($row = $result->fetch_assoc()) {
            $tables[] = $row;
        }
        return $tables;
    }
    
    /**
     * Get transaction ID for a table (current active transaction)
     */
    public function getTableTransaction($table_id) {
        // Try new method first (transaction_tables)
        $stmt = $this->db->prepare("SELECT tr.id, tr.invoice_number
            FROM transaction_tables tt
            INNER JOIN transactions tr ON tt.transaction_id = tr.id
            WHERE tt.table_id = ?
                AND tr.status = 'completed'
                AND DATE(tr.created_at) = CURDATE()
            ORDER BY tr.created_at DESC
            LIMIT 1");
        $stmt->bind_param("i", $table_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $transaction = $result->fetch_assoc();
        
        // If not found, try old method (table_id directly in transactions)
        if (!$transaction) {
            $stmt2 = $this->db->prepare("SELECT tr.id, tr.invoice_number
                FROM transactions tr
                WHERE tr.table_id = ?
                    AND tr.status = 'completed'
                    AND DATE(tr.created_at) = CURDATE()
                ORDER BY tr.created_at DESC
                LIMIT 1");
            $stmt2->bind_param("i", $table_id);
            $stmt2->execute();
            $result2 = $stmt2->get_result();
            $transaction = $result2->fetch_assoc();
        }
        
        return $transaction;
    }
}
?>

