<?php
/**
 * Transaction Class
 * Handles transaction operations
 */

class Transaction {
    private $db;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    /**
     * Create transaction
     * @param array $table_ids - Array of table IDs (can be single table or merged tables)
     * @param string $payment_status - 'paid' or 'unpaid' (default: 'paid')
     */
    public function create($items, $total, $payment_method = 'cash', $cash_received = 0, $table_ids = null, $payment_status = 'paid') {
        $this->db->autocommit(false);
        
        try {
            // Generate invoice number
            $invoice_number = 'INV-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -6));
            
            // Normalize table_ids to array
            if ($table_ids !== null && !is_array($table_ids)) {
                $table_ids = [$table_ids];
            }
            
            // Get primary table_id (first one) for backward compatibility
            $primary_table_id = !empty($table_ids) ? $table_ids[0] : null;
            
            // Calculate change only if paid
            $change = ($payment_status === 'paid') ? ($cash_received - $total) : 0;
            
            // Create transaction
            $user_id = getUserId();
            if ($primary_table_id) {
                if ($payment_status === 'paid') {
                    $stmt = $this->db->prepare("INSERT INTO transactions 
                        (invoice_number, user_id, table_id, total, payment_method, cash_received, change_amount, status, payment_status, paid_at, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, 'completed', ?, NOW(), NOW())");
                    $stmt->bind_param("siidsdds", $invoice_number, $user_id, $primary_table_id, $total, $payment_method, $cash_received, $change, $payment_status);
                } else {
                    $stmt = $this->db->prepare("INSERT INTO transactions 
                        (invoice_number, user_id, table_id, total, payment_method, cash_received, change_amount, status, payment_status, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, 'completed', ?, NOW())");
                    $stmt->bind_param("siidsdds", $invoice_number, $user_id, $primary_table_id, $total, $payment_method, $cash_received, $change, $payment_status);
                }
            } else {
                if ($payment_status === 'paid') {
                    $stmt = $this->db->prepare("INSERT INTO transactions 
                        (invoice_number, user_id, total, payment_method, cash_received, change_amount, status, payment_status, paid_at, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, 'completed', ?, NOW(), NOW())");
                    $stmt->bind_param("sidsdds", $invoice_number, $user_id, $total, $payment_method, $cash_received, $change, $payment_status);
                } else {
                    $stmt = $this->db->prepare("INSERT INTO transactions 
                        (invoice_number, user_id, total, payment_method, cash_received, change_amount, status, payment_status, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, 'completed', ?, NOW())");
                    $stmt->bind_param("sidsdds", $invoice_number, $user_id, $total, $payment_method, $cash_received, $change, $payment_status);
                }
            }
            
            $stmt->execute();
            $transaction_id = $this->db->insert_id;
            $stmt->close();
            
            // Link tables to transaction (support multiple tables for merge)
            if (!empty($table_ids) && is_array($table_ids)) {
                $table_link_stmt = $this->db->prepare("INSERT INTO transaction_tables (transaction_id, table_id, is_primary) VALUES (?, ?, ?)");
                foreach ($table_ids as $index => $table_id) {
                    $is_primary = ($index === 0) ? 1 : 0;
                    $table_link_stmt->bind_param("iii", $transaction_id, $table_id, $is_primary);
                    $table_link_stmt->execute();
                    
                    // Update table status to occupied
                    $table_stmt = $this->db->prepare("UPDATE tables SET status = 'occupied' WHERE id = ?");
                    $table_stmt->bind_param("i", $table_id);
                    $table_stmt->execute();
                    $table_stmt->close();
                }
                $table_link_stmt->close();
            }
            
            // Create transaction items (optimized - reuse prepared statements)
            $item_stmt = $this->db->prepare("INSERT INTO transaction_items 
                (transaction_id, product_id, quantity, price, subtotal, status) 
                VALUES (?, ?, ?, ?, ?, 'pending')");
            
            // Prepare stock update statement (reuse for all items - much faster)
            $stock_stmt = $this->db->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
            
            // Process all items
            foreach ($items as $item) {
                $subtotal = $item['quantity'] * $item['price'];
                
                // Insert transaction item (reuse prepared statement)
                $item_stmt->bind_param("iiidd", 
                    $transaction_id, 
                    $item['product_id'], 
                    $item['quantity'], 
                    $item['price'], 
                    $subtotal
                );
                $item_stmt->execute();
                
                // Update product stock (reuse prepared statement - no Product class overhead)
                $stock_stmt->bind_param("ii", $item['quantity'], $item['product_id']);
                $stock_stmt->execute();
            }
            
            // Close statements
            if ($item_stmt) $item_stmt->close();
            if ($stock_stmt) $stock_stmt->close();
            
            $this->db->commit();
            return $transaction_id;
            
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Transaction error: " . $e->getMessage());
            return false;
        } finally {
            $this->db->autocommit(true);
        }
    }
    
    /**
     * Get transaction by ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT t.*, u.name as cashier_name, 
            tbl.name as table_name, tbl.location as table_location
            FROM transactions t 
            LEFT JOIN users u ON t.user_id = u.id 
            LEFT JOIN tables tbl ON t.table_id = tbl.id
            WHERE t.id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $transaction = $result->fetch_assoc();
        
        // Get all merged tables
        if ($transaction) {
            $merged_tables = $this->getTransactionTables($id);
            $transaction['merged_tables'] = $merged_tables;
            if (count($merged_tables) > 1) {
                $transaction['is_merged'] = true;
                $transaction['merged_table_names'] = array_map(function($t) { return $t['name']; }, $merged_tables);
            }
        }
        
        return $transaction;
    }
    
    /**
     * Get all tables linked to a transaction
     */
    public function getTransactionTables($transaction_id) {
        $stmt = $this->db->prepare("SELECT t.*, tt.is_primary
            FROM transaction_tables tt
            INNER JOIN tables t ON tt.table_id = t.id
            WHERE tt.transaction_id = ?
            ORDER BY tt.is_primary DESC, t.name");
        $stmt->bind_param("i", $transaction_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $tables = [];
        while ($row = $result->fetch_assoc()) {
            $tables[] = $row;
        }
        return $tables;
    }
    
    /**
     * Merge tables into existing transaction
     */
    public function mergeTables($transaction_id, $table_ids) {
        if (empty($table_ids) || !is_array($table_ids)) {
            return false;
        }
        
        $this->db->autocommit(false);
        try {
            $stmt = $this->db->prepare("INSERT INTO transaction_tables (transaction_id, table_id, is_primary) VALUES (?, ?, 0)");
            
            foreach ($table_ids as $table_id) {
                // Check if already linked
                $check = $this->db->prepare("SELECT id FROM transaction_tables WHERE transaction_id = ? AND table_id = ?");
                $check->bind_param("ii", $transaction_id, $table_id);
                $check->execute();
                $result = $check->get_result();
                
                if ($result->num_rows == 0) {
                    $stmt->bind_param("ii", $transaction_id, $table_id);
                    $stmt->execute();
                    
                    // Update table status to occupied
                    $table_stmt = $this->db->prepare("UPDATE tables SET status = 'occupied' WHERE id = ?");
                    $table_stmt->bind_param("i", $table_id);
                    $table_stmt->execute();
                    $table_stmt->close();
                }
                $check->close();
            }
            
            $stmt->close();
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Merge tables error: " . $e->getMessage());
            return false;
        } finally {
            $this->db->autocommit(true);
        }
    }
    
    /**
     * Unmerge table from transaction
     */
    public function unmergeTable($transaction_id, $table_id) {
        // Check if this is the only table
        $check = $this->db->prepare("SELECT COUNT(*) as count FROM transaction_tables WHERE transaction_id = ?");
        $check->bind_param("i", $transaction_id);
        $check->execute();
        $result = $check->get_result();
        $row = $result->fetch_assoc();
        $check->close();
        
        if ($row['count'] <= 1) {
            return false; // Cannot unmerge if it's the only table
        }
        
        $stmt = $this->db->prepare("DELETE FROM transaction_tables WHERE transaction_id = ? AND table_id = ?");
        $stmt->bind_param("ii", $transaction_id, $table_id);
        $success = $stmt->execute();
        $stmt->close();
        
        if ($success) {
            // Check if table still has other transactions today
            $check_other = $this->db->prepare("SELECT COUNT(*) as count FROM transaction_tables tt
                INNER JOIN transactions tr ON tt.transaction_id = tr.id
                WHERE tt.table_id = ? 
                AND tr.status = 'completed'
                AND DATE(tr.created_at) = CURDATE()");
            $check_other->bind_param("i", $table_id);
            $check_other->execute();
            $result_other = $check_other->get_result();
            $row_other = $result_other->fetch_assoc();
            $check_other->close();
            
            // If no other transactions, set status back to available
            // Otherwise keep as occupied
            $new_status = ($row_other['count'] > 0) ? 'occupied' : 'available';
            $table_stmt = $this->db->prepare("UPDATE tables SET status = ? WHERE id = ?");
            $table_stmt->bind_param("si", $new_status, $table_id);
            $table_stmt->execute();
            $table_stmt->close();
        }
        
        return $success;
    }
    
    /**
     * Get transaction items
     */
    public function getItems($transaction_id) {
        $stmt = $this->db->prepare("SELECT ti.*, p.name as product_name, p.code as product_code 
            FROM transaction_items ti 
            LEFT JOIN products p ON ti.product_id = p.id 
            WHERE ti.transaction_id = ?
            ORDER BY 
                CASE ti.status
                    WHEN 'pending' THEN 1
                    WHEN 'preparing' THEN 2
                    WHEN 'ready' THEN 3
                    WHEN 'completed' THEN 4
                    ELSE 5
                END, ti.id");
        $stmt->bind_param("i", $transaction_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $items = [];
        while ($row = $result->fetch_assoc()) {
            $items[] = $row;
        }
        return $items;
    }
    
    /**
     * Get all transactions
     */
    public function getAll($date_from = null, $date_to = null) {
        if ($date_from && $date_to) {
            $stmt = $this->db->prepare("SELECT t.*, u.name as cashier_name 
                FROM transactions t 
                LEFT JOIN users u ON t.user_id = u.id 
                WHERE DATE(t.created_at) BETWEEN ? AND ?
                ORDER BY t.created_at DESC");
            $stmt->bind_param("ss", $date_from, $date_to);
        } else {
            $stmt = $this->db->prepare("SELECT t.*, u.name as cashier_name 
                FROM transactions t 
                LEFT JOIN users u ON t.user_id = u.id 
                ORDER BY t.created_at DESC 
                LIMIT 100");
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $transactions = [];
        while ($row = $result->fetch_assoc()) {
            $transactions[] = $row;
        }
        return $transactions;
    }
    
    /**
     * Get sales summary (only paid transactions)
     */
    public function getSalesSummary($date_from = null, $date_to = null) {
        if ($date_from && $date_to) {
            $stmt = $this->db->prepare("SELECT 
                COUNT(*) as total_transactions,
                SUM(total) as total_revenue,
                SUM(CASE WHEN payment_method = 'cash' THEN total ELSE 0 END) as cash_revenue,
                SUM(CASE WHEN payment_method = 'card' THEN total ELSE 0 END) as card_revenue,
                SUM(CASE WHEN payment_method = 'qris' THEN total ELSE 0 END) as qris_revenue
                FROM transactions 
                WHERE DATE(created_at) BETWEEN ? AND ? AND status = 'completed' AND payment_status = 'paid'");
            $stmt->bind_param("ss", $date_from, $date_to);
        } else {
            $stmt = $this->db->prepare("SELECT 
                COUNT(*) as total_transactions,
                SUM(total) as total_revenue,
                SUM(CASE WHEN payment_method = 'cash' THEN total ELSE 0 END) as cash_revenue,
                SUM(CASE WHEN payment_method = 'card' THEN total ELSE 0 END) as card_revenue,
                SUM(CASE WHEN payment_method = 'qris' THEN total ELSE 0 END) as qris_revenue
                FROM transactions 
                WHERE DATE(created_at) = CURDATE() AND status = 'completed' AND payment_status = 'paid'");
        }
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }
    
    /**
     * Complete payment for unpaid transaction
     */
    public function completePayment($transaction_id, $payment_method, $cash_received = 0) {
        $transaction = $this->getById($transaction_id);
        if (!$transaction || $transaction['payment_status'] === 'paid') {
            return false;
        }
        
        $total = floatval($transaction['total']);
        $change = ($payment_method === 'cash') ? ($cash_received - $total) : 0;
        
        $stmt = $this->db->prepare("UPDATE transactions 
            SET payment_method = ?, cash_received = ?, change_amount = ?, payment_status = 'paid', paid_at = NOW() 
            WHERE id = ?");
        $stmt->bind_param("sddi", $payment_method, $cash_received, $change, $transaction_id);
        $success = $stmt->execute();
        $stmt->close();
        
        return $success;
    }
    
    /**
     * Get unpaid transactions
     */
    public function getUnpaidTransactions($date_from = null, $date_to = null) {
        if ($date_from && $date_to) {
            $stmt = $this->db->prepare("SELECT t.*, u.name as cashier_name 
                FROM transactions t 
                LEFT JOIN users u ON t.user_id = u.id 
                WHERE DATE(t.created_at) BETWEEN ? AND ? 
                AND t.payment_status = 'unpaid' 
                AND t.status = 'completed'
                ORDER BY t.created_at DESC");
            $stmt->bind_param("ss", $date_from, $date_to);
        } else {
            $stmt = $this->db->prepare("SELECT t.*, u.name as cashier_name 
                FROM transactions t 
                LEFT JOIN users u ON t.user_id = u.id 
                WHERE t.payment_status = 'unpaid' 
                AND t.status = 'completed'
                ORDER BY t.created_at DESC");
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $transactions = [];
        while ($row = $result->fetch_assoc()) {
            $transactions[] = $row;
        }
        return $transactions;
    }
    
    /**
     * Update item status
     */
    public function updateItemStatus($item_id, $status) {
        $valid_statuses = ['pending', 'preparing', 'ready', 'completed'];
        if (!in_array($status, $valid_statuses)) {
            return false;
        }
        
        if ($status == 'completed') {
            $stmt = $this->db->prepare("UPDATE transaction_items SET status = ?, completed_at = NOW() WHERE id = ?");
        } else {
            $stmt = $this->db->prepare("UPDATE transaction_items SET status = ?, completed_at = NULL WHERE id = ?");
        }
        
        $stmt->bind_param("si", $status, $item_id);
        return $stmt->execute();
    }
    
    /**
     * Get pending orders (for kitchen)
     */
    public function getPendingOrders($date = null) {
        if (!$date) {
            $date = date('Y-m-d');
        }
        
        $stmt = $this->db->prepare("SELECT t.id, t.invoice_number, t.created_at, u.name as cashier_name,
            t.table_id, tbl.name as table_name, tbl.location as table_location,
            COUNT(ti.id) as total_items,
            SUM(CASE WHEN ti.status = 'pending' THEN 1 ELSE 0 END) as pending_count,
            SUM(CASE WHEN ti.status = 'preparing' THEN 1 ELSE 0 END) as preparing_count,
            SUM(CASE WHEN ti.status = 'ready' THEN 1 ELSE 0 END) as ready_count,
            SUM(CASE WHEN ti.status = 'completed' THEN 1 ELSE 0 END) as completed_count
            FROM transactions t
            LEFT JOIN users u ON t.user_id = u.id
            LEFT JOIN tables tbl ON t.table_id = tbl.id
            LEFT JOIN transaction_items ti ON t.id = ti.transaction_id
            WHERE DATE(t.created_at) = ?
            GROUP BY t.id
            HAVING pending_count > 0 OR preparing_count > 0 OR ready_count > 0
            ORDER BY t.created_at DESC");
        $stmt->bind_param("s", $date);
        $stmt->execute();
        $result = $stmt->get_result();
        $orders = [];
        while ($row = $result->fetch_assoc()) {
            // Get merged tables
            $merged_tables = $this->getTransactionTables($row['id']);
            $row['merged_tables'] = $merged_tables;
            if (count($merged_tables) > 1) {
                $row['is_merged'] = true;
                $row['merged_table_names'] = array_map(function($t) { return $t['name']; }, $merged_tables);
            }
            $orders[] = $row;
        }
        return $orders;
    }
}
?>

