<?php
/**
 * Global Helper Functions
 */

/**
 * Escape output to prevent XSS
 */
function escape($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Redirect to a URL
 */
function redirect($url) {
    header("Location: " . $url);
    exit();
}

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Require login
 */
function requireLogin() {
    if (!isLoggedIn()) {
        redirect(BASE_URL . 'public/login.php');
    }
}

/**
 * Get current user ID
 */
function getUserId() {
    return $_SESSION['user_id'] ?? null;
}

/**
 * Get current user data
 */
function getCurrentUser() {
    return $_SESSION['user'] ?? null;
}

/**
 * Check if user has permission
 */
function hasPermission($requiredRole) {
    $user = getCurrentUser();
    if (!$user) {
        return false;
    }
    
    // Define role hierarchy (lower index = higher privilege)
    $roles = ['admin', 'manager', 'cashier', 'waiters', 'dapur'];
    $userRoleIndex = array_search($user['role'], $roles);
    $requiredRoleIndex = array_search($requiredRole, $roles);
    
    // For waiters and dapur, use specific permissions
    if ($user['role'] == 'waiters' || $user['role'] == 'dapur') {
        return ($user['role'] == $requiredRole);
    }
    
    return $userRoleIndex !== false && $userRoleIndex <= $requiredRoleIndex;
}

/**
 * Format currency
 */
function formatCurrency($amount) {
    return 'Rp ' . number_format($amount, 0, ',', '.');
}

/**
 * Format date
 */
function formatDate($date, $format = 'd/m/Y H:i') {
    if (empty($date) || $date == '0000-00-00 00:00:00') {
        return '-';
    }
    return date($format, strtotime($date));
}

/**
 * Generate random string
 */
function generateRandomString($length = 10) {
    return substr(str_shuffle(str_repeat($x='0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length/strlen($x)) )),1,$length);
}

/**
 * Hash password
 */
function hashPassword($password) {
    return hash(HASH_ALGO, $password . 'pos_salt_2024');
}

/**
 * Verify password
 */
function verifyPassword($password, $hash) {
    return hashPassword($password) === $hash;
}

/**
 * Get flash message
 */
function getFlash($key) {
    if (isset($_SESSION['flash'][$key])) {
        $message = $_SESSION['flash'][$key];
        unset($_SESSION['flash'][$key]);
        return $message;
    }
    return null;
}

/**
 * Set flash message
 */
function setFlash($key, $message) {
    if (!isset($_SESSION['flash'])) {
        $_SESSION['flash'] = [];
    }
    $_SESSION['flash'][$key] = $message;
}

/**
 * Show alert message
 */
function showAlert($type, $message) {
    return '<div class="alert alert-' . $type . ' alert-dismissible fade show" role="alert">
        ' . escape($message) . '
        <button type="button" class="close" data-dismiss="alert" aria-label="Close">
            <span aria-hidden="true">&times;</span>
        </button>
    </div>';
}
?>

