<?php
/**
 * Setup Script
 * Run this once to set up the database and create default admin user
 */

// Hash function (same as in functions.php)
function hashPassword($password) {
    return hash('sha256', $password . 'pos_salt_2024');
}

// Connect without database first to create it
$host = 'localhost';
$username = 'root';
$password = '';
$db_name = 'pos_coffee';

$conn = new mysqli($host, $username, $password);

if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Create database if not exists
$sql = "CREATE DATABASE IF NOT EXISTS pos_coffee CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci";
if (!$conn->query($sql)) {
    die("Error creating database: " . $conn->error);
}

// Select database
$conn->select_db($db_name);

// Read SQL file
$sql = file_get_contents(__DIR__ . '/../database.sql');

// Execute SQL (split by semicolon and execute each statement)
$statements = array_filter(array_map('trim', explode(';', $sql)));

foreach ($statements as $statement) {
    $statement = trim($statement);
    if (empty($statement) || preg_match('/^(USE|CREATE DATABASE)/i', $statement)) {
        continue;
    }
    
    // Skip INSERT INTO users - we'll handle it separately below
    if (stripos($statement, 'INSERT INTO users') !== false) {
        continue;
    }
    
    if (!$conn->query($statement)) {
        // Ignore duplicate key errors
        if (strpos($conn->error, 'Duplicate') === false && strpos($conn->error, 'already exists') === false) {
            echo "Error executing statement: " . $conn->error . "<br>";
            echo "Statement: " . htmlspecialchars(substr($statement, 0, 100)) . "...<br>";
        }
    }
}

// Update passwords for default users if they already exist
$users_to_create = [
    ['username' => 'admin', 'password' => 'admin123', 'name' => 'Administrator', 'role' => 'admin'],
    ['username' => 'waiters', 'password' => 'waiters123', 'name' => 'Waiter 1', 'role' => 'waiters'],
    ['username' => 'dapur', 'password' => 'dapur123', 'name' => 'Staff Dapur', 'role' => 'dapur']
];

foreach ($users_to_create as $user_data) {
    // Check if user exists
    $stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
    $stmt->bind_param("s", $user_data['username']);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        // Update password
        $hashed_password = hashPassword($user_data['password']);
        $stmt = $conn->prepare("UPDATE users SET password = ? WHERE username = ?");
        $stmt->bind_param("ss", $hashed_password, $user_data['username']);
        $stmt->execute();
    } else {
        // Insert new user
        $hashed_password = hashPassword($user_data['password']);
        $stmt = $conn->prepare("INSERT INTO users (username, password, name, role) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("ssss", $user_data['username'], $hashed_password, $user_data['name'], $user_data['role']);
        $stmt->execute();
    }
}

$conn->close();

// Base URL for redirect
$base_url = 'http://localhost/pos/';

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Setup Complete</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 600px; margin: 50px auto; padding: 20px; }
        .success { background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; border-radius: 5px; margin: 20px 0; }
        .info { background: #d1ecf1; border: 1px solid #bee5eb; color: #0c5460; padding: 15px; border-radius: 5px; margin: 20px 0; }
        .btn { display: inline-block; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; margin-top: 10px; }
        .btn:hover { background: #0056b3; }
    </style>
</head>
<body>
    <h1>✅ Database Setup Completed!</h1>
    <div class='success'>
        <strong>Database berhasil dibuat dan diisi dengan data sample.</strong>
    </div>
    <div class='info'>
        <h3>Login Default:</h3>
        <p><strong>Admin:</strong> admin / admin123</p>
        <p><strong>Waiters:</strong> waiters / waiters123</p>
        <p><strong>Dapur:</strong> dapur / dapur123</p>
        <p><em>⚠️ Harap ganti password setelah login pertama!</em></p>
    </div>
    <a href='" . $base_url . "public/login.php' class='btn'>Go to Login →</a>
</body>
</html>";
?>

