<?php
/**
 * Helper script to create new migration file
 * 
 * Usage: php create_migration.php migration_name
 * Example: php create_migration.php add_discount_field
 */

if (php_sapi_name() !== 'cli') {
    die("This script can only be run from command line.\n");
}

if (empty($argv[1])) {
    echo "Usage: php create_migration.php migration_name\n";
    echo "Example: php create_migration.php add_discount_field\n";
    exit(1);
}

$migrationName = $argv[1];
$migrationsPath = __DIR__ . '/migrations';
$templatePath = $migrationsPath . '/.template.php';

// Validate migration name (alphanumeric and underscores only)
if (!preg_match('/^[a-z0-9_]+$/', $migrationName)) {
    echo "Error: Migration name can only contain lowercase letters, numbers, and underscores\n";
    exit(1);
}

// Generate timestamp
$timestamp = date('YmdHis');
$filename = $timestamp . '_' . $migrationName . '.php';
$filepath = $migrationsPath . '/' . $filename;

// Check if file already exists
if (file_exists($filepath)) {
    echo "Error: Migration file already exists: {$filename}\n";
    exit(1);
}

// Check if template exists
if (!file_exists($templatePath)) {
    echo "Error: Template file not found: {$templatePath}\n";
    exit(1);
}

// Read template
$template = file_get_contents($templatePath);

// Generate class name (PascalCase)
$className = '';
$parts = explode('_', $migrationName);
foreach ($parts as $part) {
    $className .= ucfirst($part);
}

// Replace template placeholders
$content = str_replace('MigrationTemplate', $className, $template);
$content = str_replace('Migration Template', $className, $content);
$content = str_replace('migration_name', $migrationName, $content);

// Add description comment
$description = ucwords(str_replace('_', ' ', $migrationName));
$content = preg_replace(
    '/\/\*\*.*?Migration Template.*?\*\//s',
    "/**\n * Migration: {$description}\n * Date: " . date('Y-m-d') . "\n */",
    $content,
    1
);

// Write file
if (file_put_contents($filepath, $content)) {
    echo "✅ Migration created successfully!\n";
    echo "File: {$filename}\n";
    echo "Class: {$className}\n";
    echo "\nNext steps:\n";
    echo "1. Edit: {$filepath}\n";
    echo "2. Implement up() and down() methods\n";
    echo "3. Run: php migrate.php run\n";
} else {
    echo "Error: Failed to create migration file\n";
    exit(1);
}






