# Database Migrations

Sistem migrasi database untuk mengelola update schema secara otomatis.

## Cara Menggunakan

### 1. Melihat Status Migrasi

**CLI:**
```bash
php migrate.php status
```

**Web:**
Akses melalui menu "Migrations" (hanya untuk admin/manager)

### 2. Menjalankan Migrasi

**CLI:**
```bash
php migrate.php run
```

**Web:**
Klik tombol "Run Pending Migrations" di halaman migrations

### 3. Rollback Migrasi

**CLI:**
```bash
php migrate.php rollback
```

**Web:**
Klik tombol "Rollback Last Batch" di halaman migrations

## Membuat Migration Baru

1. **Buat file migration:**
   ```bash
   cp migrations/.template.php migrations/YYYYMMDDHHMMSS_descriptive_name.php
   ```
   
   Contoh:
   ```bash
   cp migrations/.template.php migrations/20240115120000_add_discount_field.php
   ```

2. **Edit file migration:**
   - Rename class sesuai dengan nama file (PascalCase)
   - Implementasikan method `up()` untuk perubahan forward
   - Implementasikan method `down()` untuk rollback

3. **Contoh Migration:**

```php
<?php
class AddDiscountField extends Migration {
    
    public function up() {
        if (!$this->columnExists('transactions', 'discount')) {
            $this->execute("ALTER TABLE transactions 
                ADD COLUMN discount DECIMAL(10,2) DEFAULT 0 AFTER total");
        }
    }
    
    public function down() {
        if ($this->columnExists('transactions', 'discount')) {
            $this->execute("ALTER TABLE transactions DROP COLUMN discount");
        }
    }
}
```

## Helper Methods

Migration class menyediakan helper methods:

- `columnExists($table, $column)` - Check jika column ada
- `tableExists($table)` - Check jika table ada
- `execute($sql)` - Execute SQL query dengan error handling

## Best Practices

1. **Naming Convention:**
   - Format: `YYYYMMDDHHMMSS_descriptive_name.php`
   - Class name: PascalCase dari descriptive_name
   - Contoh: `20240115120000_add_discount.php` → `AddDiscount`

2. **Idempotent:**
   - Selalu check jika perubahan sudah ada sebelum execute
   - Gunakan `columnExists()` atau `tableExists()` sebelum ALTER/CREATE

3. **Rollback:**
   - Selalu implementasikan `down()` method untuk rollback
   - Pastikan rollback bisa mengembalikan ke state sebelum migration

4. **Testing:**
   - Test migration dengan run dan rollback
   - Pastikan tidak ada data loss

## Struktur File

```
migrations/
├── README.md                    # Dokumentasi ini
├── .template.php                # Template untuk migration baru
├── 20240101000001_*.php         # Migration files (sorted by timestamp)
├── 20240101000002_*.php
└── ...
```

## Migration Tracking

Sistem menggunakan table `migrations` untuk tracking:
- `migration_name` - Nama file migration (tanpa .php)
- `batch` - Batch number (untuk rollback)
- `executed_at` - Waktu execution
- `execution_time` - Durasi execution (detik)

## Troubleshooting

### Migration gagal
1. Check error message di console/web
2. Pastikan SQL syntax benar
3. Pastikan tidak ada conflict dengan existing schema
4. Check database connection

### Rollback gagal
1. Pastikan `down()` method sudah diimplementasikan
2. Check jika perubahan sudah bisa di-rollback (misal: data sudah ada)
3. Manual rollback jika perlu

### Migration sudah dijalankan tapi masih pending
- Check table `migrations` di database
- Pastikan nama migration file sesuai dengan yang di database
- Hapus record di table `migrations` jika perlu re-run (tidak disarankan)






