<?php
/**
 * API Endpoint untuk Sync Offline Transactions
 * Menerima transaksi dari client dan menyimpannya ke database
 */

header('Content-Type: application/json');
// CORS headers - allow same origin only for security
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header('Access-Control-Allow-Origin: ' . $_SERVER['HTTP_ORIGIN']);
}
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Credentials: true');

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/functions.php';

// Check if user is logged in
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid JSON']);
    exit;
}

// Validate required fields
if (empty($data['items']) || !is_array($data['items']) || count($data['items']) === 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Items required']);
    exit;
}

if (empty($data['total']) || $data['total'] <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid total']);
    exit;
}

try {
    $transaction = new Transaction();
    
    // Extract data
    $items = $data['items'];
    $total = floatval($data['total']);
    $payment_method = $data['payment_method'] ?? 'cash';
    $cash_received = floatval($data['cash_received'] ?? $total);
    $table_ids = $data['table_ids'] ?? null;
    $payment_status = $data['payment_status'] ?? ($data['pay_later'] ? 'unpaid' : 'paid');
    
    // Check for duplicate (idempotent - check by local_id or invoice_number)
    $local_id = $data['local_id'] ?? null;
    $invoice_number = $data['invoice_number'] ?? null;
    
    if ($invoice_number) {
        // Check if transaction already exists
        $db = new Database();
        $conn = $db->getConnection();
        $checkStmt = $conn->prepare("SELECT id FROM transactions WHERE invoice_number = ?");
        $checkStmt->bind_param("s", $invoice_number);
        $checkStmt->execute();
        $result = $checkStmt->get_result();
        
        if ($result->num_rows > 0) {
            $existing = $result->fetch_assoc();
            $checkStmt->close();
            
            // Transaction already exists, return success with existing ID
            echo json_encode([
                'success' => true,
                'transaction_id' => $existing['id'],
                'message' => 'Transaction already synced',
                'duplicate' => true
            ]);
            exit;
        }
        $checkStmt->close();
    }
    
    // Create transaction
    $transaction_id = $transaction->create(
        $items,
        $total,
        $payment_method,
        $cash_received,
        $table_ids,
        $payment_status
    );
    
    if ($transaction_id) {
        echo json_encode([
            'success' => true,
            'transaction_id' => $transaction_id,
            'message' => 'Transaction synced successfully'
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to create transaction'
        ]);
    }
    
} catch (Exception $e) {
    error_log("Sync transaction error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server error: ' . $e->getMessage()
    ]);
}
?>

