/**
 * IndexedDB Wrapper untuk Offline Transaction Storage
 * Menyimpan transaksi, produk, dan sync queue
 */

class OfflineDB {
  constructor() {
    this.dbName = 'POS_OfflineDB';
    this.version = 1;
    this.db = null;
  }

  /**
   * Initialize database
   */
  async init() {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.version);

      request.onerror = () => {
        console.error('[OfflineDB] Failed to open database');
        reject(request.error);
      };

      request.onsuccess = () => {
        this.db = request.result;
        console.log('[OfflineDB] Database opened successfully');
        resolve(this.db);
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;

        // Store untuk transaksi offline
        if (!db.objectStoreNames.contains('transactions')) {
          const transactionStore = db.createObjectStore('transactions', {
            keyPath: 'id',
            autoIncrement: true
          });
          transactionStore.createIndex('sync_status', 'sync_status', { unique: false });
          transactionStore.createIndex('created_at', 'created_at', { unique: false });
          transactionStore.createIndex('invoice_number', 'invoice_number', { unique: true });
        }

        // Store untuk transaction items
        if (!db.objectStoreNames.contains('transaction_items')) {
          const itemsStore = db.createObjectStore('transaction_items', {
            keyPath: 'id',
            autoIncrement: true
          });
          itemsStore.createIndex('transaction_id', 'transaction_id', { unique: false });
        }

        // Store untuk produk snapshot (untuk referensi offline)
        if (!db.objectStoreNames.contains('products_snapshot')) {
          const productsStore = db.createObjectStore('products_snapshot', {
            keyPath: 'id'
          });
          productsStore.createIndex('code', 'code', { unique: true });
          productsStore.createIndex('category_id', 'category_id', { unique: false });
        }

        // Store untuk sync queue
        if (!db.objectStoreNames.contains('sync_queue')) {
          const syncStore = db.createObjectStore('sync_queue', {
            keyPath: 'id',
            autoIncrement: true
          });
          syncStore.createIndex('transaction_id', 'transaction_id', { unique: false });
          syncStore.createIndex('status', 'status', { unique: false });
          syncStore.createIndex('created_at', 'created_at', { unique: false });
        }

        // Store untuk categories snapshot
        if (!db.objectStoreNames.contains('categories_snapshot')) {
          db.createObjectStore('categories_snapshot', {
            keyPath: 'id'
          });
        }

        console.log('[OfflineDB] Database schema created');
      };
    });
  }

  /**
   * Save transaction offline
   */
  async saveTransaction(transactionData) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transactions', 'transaction_items'], 'readwrite');

      // Generate local invoice number jika belum ada
      if (!transactionData.invoice_number) {
        transactionData.invoice_number = 'OFFLINE-' + Date.now() + '-' + Math.random().toString(36).substr(2, 6).toUpperCase();
      }

      // Set sync status
      transactionData.sync_status = 'PENDING_SYNC';
      transactionData.created_at = new Date().toISOString();
      transactionData.local_id = transactionData.id || 'local_' + Date.now();

      const transactionStore = transaction.objectStore('transactions');
      const request = transactionStore.add(transactionData);

      request.onsuccess = async () => {
        const transactionId = request.result;
        transactionData.id = transactionId;

        // Save transaction items
        if (transactionData.items && transactionData.items.length > 0) {
          const itemsStore = transaction.objectStore('transaction_items');
          const itemsPromises = transactionData.items.map(item => {
            return new Promise((resolveItem, rejectItem) => {
              const itemData = {
                transaction_id: transactionId,
                product_id: item.product_id,
                product_name: item.product_name,
                product_code: item.product_code,
                quantity: item.quantity,
                price: item.price,
                subtotal: item.subtotal
              };
              const itemRequest = itemsStore.add(itemData);
              itemRequest.onsuccess = () => resolveItem(itemRequest.result);
              itemRequest.onerror = () => rejectItem(itemRequest.error);
            });
          });

          try {
            await Promise.all(itemsPromises);
          } catch (error) {
            reject(error);
            return;
          }
        }

        // Add to sync queue
        await this.addToSyncQueue(transactionId, transactionData);

        resolve(transactionId);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Get all pending transactions
   */
  async getPendingTransactions() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transactions'], 'readonly');
      const store = transaction.objectStore('transactions');
      const index = store.index('sync_status');
      const request = index.getAll('PENDING_SYNC');

      request.onsuccess = async () => {
        const transactions = request.result;
        
        // Get items for each transaction
        const transactionsWithItems = await Promise.all(
          transactions.map(async (tx) => {
            tx.items = await this.getTransactionItems(tx.id);
            return tx;
          })
        );

        resolve(transactionsWithItems);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Get transaction items
   */
  async getTransactionItems(transactionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transaction_items'], 'readonly');
      const store = transaction.objectStore('transaction_items');
      const index = store.index('transaction_id');
      const request = index.getAll(transactionId);

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Update transaction sync status
   */
  async updateSyncStatus(transactionId, status, serverId = null) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transactions'], 'readwrite');
      const store = transaction.objectStore('transactions');
      const request = store.get(transactionId);

      request.onsuccess = () => {
        const data = request.result;
        if (data) {
          data.sync_status = status;
          if (serverId) {
            data.server_id = serverId;
          }
          
          const updateRequest = store.put(data);
          updateRequest.onsuccess = () => resolve();
          updateRequest.onerror = () => reject(updateRequest.error);
        } else {
          reject(new Error('Transaction not found'));
        }
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Save products snapshot
   */
  async saveProductsSnapshot(products) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['products_snapshot'], 'readwrite');
      const store = transaction.objectStore('products_snapshot');
      
      // Clear existing
      store.clear();

      // Add all products
      const promises = products.map(product => {
        return new Promise((resolveItem, rejectItem) => {
          const request = store.add(product);
          request.onsuccess = () => resolveItem();
          request.onerror = () => rejectItem(request.error);
        });
      });

      Promise.all(promises)
        .then(() => {
          console.log('[OfflineDB] Products snapshot saved:', products.length);
          resolve();
        })
        .catch(reject);
    });
  }

  /**
   * Get products snapshot
   */
  async getProductsSnapshot() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['products_snapshot'], 'readonly');
      const store = transaction.objectStore('products_snapshot');
      const request = store.getAll();

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Save categories snapshot
   */
  async saveCategoriesSnapshot(categories) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['categories_snapshot'], 'readwrite');
      const store = transaction.objectStore('categories_snapshot');
      
      store.clear();

      const promises = categories.map(category => {
        return new Promise((resolveItem, rejectItem) => {
          const request = store.add(category);
          request.onsuccess = () => resolveItem();
          request.onerror = () => rejectItem(request.error);
        });
      });

      Promise.all(promises)
        .then(() => {
          console.log('[OfflineDB] Categories snapshot saved:', categories.length);
          resolve();
        })
        .catch(reject);
    });
  }

  /**
   * Get categories snapshot
   */
  async getCategoriesSnapshot() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['categories_snapshot'], 'readonly');
      const store = transaction.objectStore('categories_snapshot');
      const request = store.getAll();

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Add to sync queue
   */
  async addToSyncQueue(transactionId, transactionData) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sync_queue'], 'readwrite');
      const store = transaction.objectStore('sync_queue');
      
      const queueItem = {
        transaction_id: transactionId,
        transaction_data: transactionData,
        status: 'PENDING',
        created_at: new Date().toISOString(),
        retry_count: 0
      };

      const request = store.add(queueItem);
      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }

  /**
   * Get sync queue
   */
  async getSyncQueue() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sync_queue'], 'readonly');
      const store = transaction.objectStore('sync_queue');
      const index = store.index('status');
      const request = index.getAll('PENDING');

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Update sync queue status
   */
  async updateSyncQueueStatus(queueId, status) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sync_queue'], 'readwrite');
      const store = transaction.objectStore('sync_queue');
      const request = store.get(queueId);

      request.onsuccess = () => {
        const item = request.result;
        if (item) {
          item.status = status;
          if (status === 'FAILED') {
            item.retry_count = (item.retry_count || 0) + 1;
          }
          
          const updateRequest = store.put(item);
          updateRequest.onsuccess = () => resolve();
          updateRequest.onerror = () => reject(updateRequest.error);
        } else {
          reject(new Error('Queue item not found'));
        }
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Delete synced transaction
   */
  async deleteSyncedTransaction(transactionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transactions', 'transaction_items', 'sync_queue'], 'readwrite');
      
      // Delete transaction
      const txStore = transaction.objectStore('transactions');
      const txRequest = txStore.delete(transactionId);
      
      // Delete items
      const itemsStore = transaction.objectStore('transaction_items');
      const itemsIndex = itemsStore.index('transaction_id');
      const itemsRequest = itemsIndex.openCursor(IDBKeyRange.only(transactionId));
      
      itemsRequest.onsuccess = (event) => {
        const cursor = event.target.result;
        if (cursor) {
          cursor.delete();
          cursor.continue();
        }
      };

      // Delete from sync queue
      const queueStore = transaction.objectStore('sync_queue');
      const queueIndex = queueStore.index('transaction_id');
      const queueRequest = queueIndex.openCursor(IDBKeyRange.only(transactionId));
      
      queueRequest.onsuccess = (event) => {
        const cursor = event.target.result;
        if (cursor) {
          cursor.delete();
          cursor.continue();
        }
      };

      transaction.oncomplete = () => {
        resolve();
      };

      transaction.onerror = () => {
        reject(transaction.error);
      };
    });
  }

  /**
   * Get transaction count by status
   */
  async getTransactionCount() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transactions'], 'readonly');
      const store = transaction.objectStore('transactions');
      const request = store.count();

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Get pending sync count
   */
  async getPendingSyncCount() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['transactions'], 'readonly');
      const store = transaction.objectStore('transactions');
      const index = store.index('sync_status');
      const request = index.count('PENDING_SYNC');

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }
}

// Export singleton instance
const offlineDB = new OfflineDB();

