/**
 * Offline Status Indicator Component
 * Menampilkan status koneksi dan sync di UI
 */

class OfflineStatusIndicator {
  constructor() {
    this.status = 'ONLINE';
    this.pendingCount = 0;
    this.syncStatus = 'IDLE';
    this.init();
  }

  async init() {
    // Create status indicator element
    this.createIndicator();
    
    // Listen for online/offline events
    window.addEventListener('online', () => {
      this.updateStatus('ONLINE');
      this.updateIndicator();
    });

    window.addEventListener('offline', () => {
      this.updateStatus('OFFLINE');
      this.updateIndicator();
    });

    // Listen for sync status updates
    window.addEventListener('syncStatusUpdate', (event) => {
      this.syncStatus = event.detail.status;
      this.updateIndicator();
    });

    // Listen for sync complete
    window.addEventListener('syncComplete', (event) => {
      this.updatePendingCount();
    });

    // Initial status check
    this.updateStatus(navigator.onLine ? 'ONLINE' : 'OFFLINE');
    await this.updatePendingCount();
    this.updateIndicator();

    // Periodic update
    setInterval(() => {
      this.updatePendingCount();
    }, 10000); // Every 10 seconds
  }

  createIndicator() {
    // Create status bar
    const statusBar = document.createElement('div');
    statusBar.id = 'offline-status-bar';
    statusBar.className = 'offline-status-bar';
    
    statusBar.innerHTML = `
      <div class="status-content">
        <div class="status-indicator">
          <span class="status-icon" id="statusIcon">●</span>
          <span class="status-text" id="statusText">Online</span>
        </div>
        <div class="sync-info" id="syncInfo" style="display: none;">
          <span class="sync-text" id="syncText"></span>
          <button class="sync-btn" id="syncBtn" onclick="syncEngine.manualSync()">Sync</button>
        </div>
      </div>
    `;

    // Insert at top of body
    document.body.insertBefore(statusBar, document.body.firstChild);

    // Add styles
    this.addStyles();
  }

  updateStatus(status) {
    this.status = status;
  }

  async updatePendingCount() {
    try {
      if (offlineDB && offlineDB.db) {
        this.pendingCount = await offlineDB.getPendingSyncCount();
      }
    } catch (error) {
      console.error('[StatusIndicator] Error updating pending count:', error);
    }
  }

  updateIndicator() {
    const statusBar = document.getElementById('offline-status-bar');
    if (!statusBar) return;

    const statusIcon = document.getElementById('statusIcon');
    const statusText = document.getElementById('statusText');
    const syncInfo = document.getElementById('syncInfo');
    const syncText = document.getElementById('syncText');
    const syncBtn = document.getElementById('syncBtn');

    // Update status
    if (this.status === 'ONLINE') {
      statusBar.className = 'offline-status-bar status-online';
      statusIcon.textContent = '●';
      statusIcon.className = 'status-icon status-online-icon';
      statusText.textContent = 'Online';
    } else {
      statusBar.className = 'offline-status-bar status-offline';
      statusIcon.textContent = '●';
      statusIcon.className = 'status-icon status-offline-icon';
      statusText.textContent = 'Offline';
    }

    // Update sync info
    if (this.pendingCount > 0) {
      syncInfo.style.display = 'flex';
      
      if (this.syncStatus === 'SYNCING') {
        syncText.textContent = `Menyinkronkan ${this.pendingCount} transaksi...`;
        syncBtn.disabled = true;
        syncBtn.textContent = 'Sync...';
      } else if (this.syncStatus === 'SYNC_ERROR') {
        syncText.textContent = `Gagal sinkronisasi ${this.pendingCount} transaksi`;
        syncBtn.disabled = false;
        syncBtn.textContent = 'Coba Lagi';
      } else {
        syncText.textContent = `${this.pendingCount} transaksi menunggu sinkronisasi`;
        syncBtn.disabled = false;
        syncBtn.textContent = 'Sync';
      }
    } else {
      syncInfo.style.display = 'none';
    }
  }

  addStyles() {
    const style = document.createElement('style');
    style.textContent = `
      .offline-status-bar {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        z-index: 10000;
        padding: 0.5rem 1rem;
        font-size: 0.875rem;
        font-weight: 600;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        transition: all 0.3s ease;
      }

      .offline-status-bar.status-online {
        background: linear-gradient(135deg, #10B981, #059669);
        color: white;
      }

      .offline-status-bar.status-offline {
        background: linear-gradient(135deg, #F59E0B, #D97706);
        color: white;
        animation: pulse 2s infinite;
      }

      @keyframes pulse {
        0%, 100% {
          opacity: 1;
        }
        50% {
          opacity: 0.8;
        }
      }

      .status-content {
        display: flex;
        justify-content: space-between;
        align-items: center;
        max-width: 1200px;
        margin: 0 auto;
      }

      .status-indicator {
        display: flex;
        align-items: center;
        gap: 0.5rem;
      }

      .status-icon {
        font-size: 0.75rem;
        animation: blink 2s infinite;
      }

      .status-online-icon {
        color: #D1FAE5;
      }

      .status-offline-icon {
        color: #FEF3C7;
      }

      @keyframes blink {
        0%, 100% {
          opacity: 1;
        }
        50% {
          opacity: 0.5;
        }
      }

      .sync-info {
        display: flex;
        align-items: center;
        gap: 0.75rem;
      }

      .sync-text {
        font-size: 0.8125rem;
      }

      .sync-btn {
        background: rgba(255, 255, 255, 0.2);
        border: 1px solid rgba(255, 255, 255, 0.3);
        color: white;
        padding: 0.25rem 0.75rem;
        border-radius: 4px;
        cursor: pointer;
        font-size: 0.75rem;
        font-weight: 600;
        transition: all 0.2s;
      }

      .sync-btn:hover:not(:disabled) {
        background: rgba(255, 255, 255, 0.3);
        transform: scale(1.05);
      }

      .sync-btn:disabled {
        opacity: 0.6;
        cursor: not-allowed;
      }

      /* Adjust body padding when status bar is visible */
      body {
        padding-top: 0;
      }

      body:has(.offline-status-bar) {
        padding-top: 40px;
      }
    `;
    document.head.appendChild(style);
  }
}

// Initialize status indicator
let statusIndicator;
window.addEventListener('load', () => {
  statusIndicator = new OfflineStatusIndicator();
});

