/**
 * Sync Engine untuk sinkronisasi transaksi offline ke server
 * Menggunakan Background Sync API dan manual retry
 */

class SyncEngine {
  constructor() {
    this.syncInProgress = false;
    this.syncInterval = null;
    this.baseUrl = window.location.origin + '/pos/public';
    this.retryDelay = 5000; // 5 seconds
    this.maxRetries = 3;
  }

  /**
   * Initialize sync engine
   */
  async init() {
    // Check if online
    if (navigator.onLine) {
      await this.syncPendingTransactions();
    }

    // Listen for online/offline events
    window.addEventListener('online', () => {
      console.log('[SyncEngine] Online - Starting sync');
      this.syncPendingTransactions();
    });

    window.addEventListener('offline', () => {
      console.log('[SyncEngine] Offline - Pausing sync');
    });

    // Periodic sync check (every 30 seconds when online)
    this.syncInterval = setInterval(() => {
      if (navigator.onLine && !this.syncInProgress) {
        this.syncPendingTransactions();
      }
    }, 30000);

    // Register background sync
    if ('serviceWorker' in navigator && 'sync' in window.ServiceWorkerRegistration.prototype) {
      navigator.serviceWorker.ready.then(registration => {
        registration.sync.register('sync-transactions').catch(err => {
          console.warn('[SyncEngine] Background sync registration failed:', err);
        });
      });
    }
  }

  /**
   * Sync pending transactions to server
   */
  async syncPendingTransactions() {
    if (this.syncInProgress) {
      console.log('[SyncEngine] Sync already in progress');
      return;
    }

    if (!navigator.onLine) {
      console.log('[SyncEngine] Offline - Cannot sync');
      return;
    }

    this.syncInProgress = true;
    
    try {
      // Get pending transactions from IndexedDB
      const pendingTransactions = await offlineDB.getPendingTransactions();
      
      if (pendingTransactions.length === 0) {
        console.log('[SyncEngine] No pending transactions to sync');
        this.syncInProgress = false;
        return;
      }

      console.log(`[SyncEngine] Syncing ${pendingTransactions.length} pending transactions`);

      // Update UI
      this.updateSyncStatus('SYNCING', pendingTransactions.length);

      // Sync each transaction
      const results = await Promise.allSettled(
        pendingTransactions.map(tx => this.syncSingleTransaction(tx))
      );

      // Count successes and failures
      const successful = results.filter(r => r.status === 'fulfilled').length;
      const failed = results.filter(r => r.status === 'rejected').length;

      console.log(`[SyncEngine] Sync complete: ${successful} successful, ${failed} failed`);

      // Update UI
      if (failed === 0) {
        this.updateSyncStatus('SYNCED', 0);
      } else {
        this.updateSyncStatus('SYNC_ERROR', failed);
      }

      // Trigger event
      window.dispatchEvent(new CustomEvent('syncComplete', {
        detail: { successful, failed, total: pendingTransactions.length }
      }));

    } catch (error) {
      console.error('[SyncEngine] Sync error:', error);
      this.updateSyncStatus('SYNC_ERROR', 0);
    } finally {
      this.syncInProgress = false;
    }
  }

  /**
   * Sync single transaction
   */
  async syncSingleTransaction(transaction) {
    try {
      // Prepare transaction data for server
      const transactionData = {
        items: transaction.items || [],
        total: transaction.total,
        payment_method: transaction.payment_method || 'cash',
        cash_received: transaction.cash_received || transaction.total,
        table_ids: transaction.table_ids || null,
        payment_status: transaction.payment_status || 'paid',
        pay_later: transaction.payment_status === 'unpaid',
        local_id: transaction.local_id,
        invoice_number: transaction.invoice_number
      };

      // Send to server
      const response = await fetch(`${this.baseUrl}/api/sync-transaction.php`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(transactionData),
        credentials: 'same-origin'
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();

      if (result.success) {
        // Update sync status
        await offlineDB.updateSyncStatus(transaction.id, 'SYNCED', result.transaction_id);
        
        // Delete from local DB after successful sync (optional - bisa di-keep untuk history)
        // await offlineDB.deleteSyncedTransaction(transaction.id);
        
        console.log(`[SyncEngine] Transaction ${transaction.invoice_number} synced successfully`);
        return { success: true, transaction_id: result.transaction_id };
      } else {
        throw new Error(result.error || 'Sync failed');
      }

    } catch (error) {
      console.error(`[SyncEngine] Failed to sync transaction ${transaction.invoice_number}:`, error);
      
      // Update retry count
      const queue = await offlineDB.getSyncQueue();
      const queueItem = queue.find(q => q.transaction_id === transaction.id);
      if (queueItem) {
        if (queueItem.retry_count < this.maxRetries) {
          await offlineDB.updateSyncQueueStatus(queueItem.id, 'PENDING');
        } else {
          await offlineDB.updateSyncQueueStatus(queueItem.id, 'FAILED');
        }
      }

      throw error;
    }
  }

  /**
   * Update sync status in UI
   */
  updateSyncStatus(status, count = 0) {
    const event = new CustomEvent('syncStatusUpdate', {
      detail: { status, count }
    });
    window.dispatchEvent(event);
  }

  /**
   * Manual sync trigger
   */
  async manualSync() {
    if (!navigator.onLine) {
      alert('Tidak ada koneksi internet. Sinkronisasi akan dilakukan otomatis saat online.');
      return;
    }

    await this.syncPendingTransactions();
  }

  /**
   * Stop sync engine
   */
  stop() {
    if (this.syncInterval) {
      clearInterval(this.syncInterval);
      this.syncInterval = null;
    }
  }
}

// Export singleton instance
const syncEngine = new SyncEngine();

