<?php
/**
 * Dashboard
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();

$currentUser = getCurrentUser();
$isWaiter = ($currentUser['role'] == 'waiters');

$pageTitle = 'Dashboard';

$transaction = new Transaction();
$summary = $transaction->getSalesSummary();

// Get today's transactions
$today = date('Y-m-d');
$transactions = $transaction->getAll($today, $today);
$transactions = array_slice($transactions, 0, 10); // Latest 10

// Get unpaid transactions count
$unpaid_transactions = $transaction->getUnpaidTransactions($today, $today);
$unpaid_count = count($unpaid_transactions);
$unpaid_total = array_sum(array_column($unpaid_transactions, 'total'));

// For waiters: get ready orders and table stats
$ready_orders_count = 0;
$occupied_tables_count = 0;
$total_ready_items = 0;
if ($isWaiter) {
    $table = new Table();
    $tables_with_ready = $table->getTablesWithReadyOrders();
    $ready_orders_count = count($tables_with_ready);
    
    $all_tables = $table->getAll();
    foreach ($all_tables as $tbl) {
        if ($tbl['status'] == 'occupied') {
            $occupied_tables_count++;
        }
        $table_status = $table->getTableStatus($tbl['id']);
        if ($table_status && isset($table_status['ready_items'])) {
            $total_ready_items += $table_status['ready_items'];
        }
    }
}

include __DIR__ . '/includes/header.php';
?>

<?php if ($isWaiter): ?>
<!-- Waiter Dashboard -->
<div class="waiter-dashboard">
    <div class="waiter-header">
        <div class="waiter-greeting">
            <h1>🍽️ Selamat Bekerja, <?php echo escape($currentUser['name']); ?>!</h1>
            <p>Dashboard Waiter - <?php echo formatDate($today, 'd F Y'); ?></p>
        </div>
        <div class="waiter-time">
            <div class="time-display" id="current-time"></div>
        </div>
    </div>

    <!-- Priority Alert: Ready Orders -->
    <?php if ($ready_orders_count > 0 || $total_ready_items > 0): ?>
    <div class="waiter-alert-card">
        <div class="alert-icon">🚨</div>
        <div class="alert-content">
            <h2>Pesanan Siap Diantar!</h2>
            <p><strong><?php echo $ready_orders_count; ?> meja</strong> dengan <strong><?php echo $total_ready_items; ?> item</strong> siap diantar</p>
        </div>
        <a href="<?php echo BASE_URL; ?>public/waiter.php" class="btn btn-lg waiter-action-btn">
            Lihat Detail →
        </a>
    </div>
    <?php endif; ?>

    <!-- Waiter Stats Grid -->
    <div class="waiter-stats-grid">
        <div class="waiter-stat-card stat-ready">
            <div class="stat-icon-wrapper">
                <div class="stat-icon">🍽️</div>
            </div>
            <div class="stat-content">
                <h3>Pesanan Siap</h3>
                <p class="stat-value"><?php echo $ready_orders_count; ?></p>
                <p class="stat-label">meja</p>
            </div>
            <div class="stat-badge"><?php echo $total_ready_items; ?> item</div>
        </div>

        <div class="waiter-stat-card stat-occupied">
            <div class="stat-icon-wrapper">
                <div class="stat-icon">🪑</div>
            </div>
            <div class="stat-content">
                <h3>Meja Terisi</h3>
                <p class="stat-value"><?php echo $occupied_tables_count; ?></p>
                <p class="stat-label">meja aktif</p>
            </div>
        </div>

        <div class="waiter-stat-card stat-today">
            <div class="stat-icon-wrapper">
                <div class="stat-icon">📊</div>
            </div>
            <div class="stat-content">
                <h3>Transaksi Hari Ini</h3>
                <p class="stat-value"><?php echo number_format($summary['total_transactions'] ?? 0); ?></p>
                <p class="stat-label">transaksi</p>
            </div>
        </div>

        <div class="waiter-stat-card stat-revenue">
            <div class="stat-icon-wrapper">
                <div class="stat-icon">💰</div>
            </div>
            <div class="stat-content">
                <h3>Pendapatan Hari Ini</h3>
                <p class="stat-value"><?php echo formatCurrency($summary['total_revenue'] ?? 0); ?></p>
            </div>
        </div>
    </div>

    <!-- Quick Actions for Waiters -->
    <div class="waiter-actions-section">
        <h2 class="section-title">⚡ Akses Cepat</h2>
        <div class="waiter-quick-actions">
            <a href="<?php echo BASE_URL; ?>public/waiter.php" class="waiter-action-card action-primary">
                <div class="action-icon">🍽️</div>
                <h3>Status Meja & Pesanan</h3>
                <p>Lihat pesanan siap dan kelola meja</p>
                <?php if ($ready_orders_count > 0): ?>
                    <span class="action-badge"><?php echo $ready_orders_count; ?> baru</span>
                <?php endif; ?>
            </a>

            <a href="<?php echo BASE_URL; ?>public/kitchen.php" class="waiter-action-card action-secondary">
                <div class="action-icon">👨‍🍳</div>
                <h3>Status Dapur</h3>
                <p>Pantau progress pesanan di dapur</p>
            </a>

            <a href="<?php echo BASE_URL; ?>public/transactions.php" class="waiter-action-card action-info">
                <div class="action-icon">📋</div>
                <h3>Daftar Transaksi</h3>
                <p>Lihat semua transaksi hari ini</p>
            </a>

            <?php if ($unpaid_count > 0): ?>
            <a href="<?php echo BASE_URL; ?>public/transactions.php?filter_payment=unpaid" class="waiter-action-card action-warning">
                <div class="action-icon">⚠️</div>
                <h3>Belum Dibayar</h3>
                <p><?php echo $unpaid_count; ?> transaksi</p>
                <span class="action-badge"><?php echo formatCurrency($unpaid_total); ?></span>
            </a>
            <?php endif; ?>
        </div>
    </div>

    <!-- Recent Transactions -->
    <div class="waiter-recent-section">
        <div class="waiter-card">
            <div class="waiter-card-header">
                <h2>📋 Transaksi Terbaru</h2>
                <a href="<?php echo BASE_URL; ?>public/transactions.php" class="btn btn-sm waiter-btn-outline">Lihat Semua</a>
            </div>
            <div class="waiter-card-body">
                <div class="waiter-table-responsive">
                    <table class="waiter-table">
                        <thead>
                            <tr>
                                <th>Invoice</th>
                                <th>Meja</th>
                                <th>Total</th>
                                <th>Waktu</th>
                                <th>Status</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($transactions)): ?>
                                <tr>
                                    <td colspan="6" class="text-center">Belum ada transaksi hari ini</td>
                                </tr>
                            <?php else: ?>
                            <?php foreach ($transactions as $t): 
                                $payment_status = $t['payment_status'] ?? 'paid';
                                // Get table name from merged tables or old table_id
                                $table_name = 'Take Away';
                                $merged_tables = $transaction->getTransactionTables($t['id']);
                                if (!empty($merged_tables)) {
                                    $table_names = array_map(function($tbl) { return $tbl['name']; }, $merged_tables);
                                    $table_name = count($table_names) > 1 ? implode(', ', $table_names) . ' (Gabungan)' : $table_names[0];
                                } elseif (!empty($t['table_id'])) {
                                    $table_obj = new Table();
                                    $table_info = $table_obj->getById($t['table_id']);
                                    $table_name = $table_info ? $table_info['name'] : 'Take Away';
                                }
                            ?>
                                <tr class="<?php echo $payment_status === 'unpaid' ? 'waiter-table-warning' : ''; ?>">
                                    <td>
                                        <strong><?php echo escape($t['invoice_number']); ?></strong>
                                    </td>
                                    <td>
                                        <span class="waiter-badge waiter-badge-info"><?php echo escape($table_name); ?></span>
                                    </td>
                                    <td><strong><?php echo formatCurrency($t['total']); ?></strong></td>
                                    <td><?php echo formatDate($t['created_at'], 'H:i'); ?></td>
                                    <td>
                                        <?php if ($payment_status === 'unpaid'): ?>
                                            <span class="waiter-badge waiter-badge-warning">Belum Bayar</span>
                                        <?php else: ?>
                                            <span class="waiter-badge waiter-badge-success">Lunas</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <a href="<?php echo BASE_URL; ?>public/transaction_detail.php?id=<?php echo $t['id']; ?>" class="btn btn-xs waiter-btn-primary">Detail</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Waiter Dashboard Styles */
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.waiter-dashboard {
    animation: fadeIn 0.5s ease;
}

.waiter-header {
    background: linear-gradient(135deg, #FF6B35 0%, #F7931E 50%, #FFC947 100%);
    padding: 2.5rem;
    border-radius: 20px;
    margin-bottom: 2rem;
    box-shadow: 0 8px 24px rgba(255, 107, 53, 0.3);
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1.5rem;
}

.waiter-greeting h1 {
    color: white;
    font-size: 2.5rem;
    margin: 0 0 0.5rem 0;
    text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
    font-weight: 700;
}

.waiter-greeting p {
    color: rgba(255,255,255,0.95);
    font-size: 1.1rem;
    margin: 0;
}

.waiter-time {
    background: rgba(255,255,255,0.2);
    padding: 1rem 1.5rem;
    border-radius: 12px;
    backdrop-filter: blur(10px);
}

.time-display {
    color: white;
    font-size: 1.5rem;
    font-weight: 600;
    font-family: 'Courier New', monospace;
}

.waiter-alert-card {
    background: linear-gradient(135deg, #FF4757 0%, #FF6348 100%);
    padding: 2rem;
    border-radius: 16px;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 2rem;
    box-shadow: 0 8px 24px rgba(255, 71, 87, 0.4);
    animation: pulse-alert 2s infinite;
    color: white;
}

@keyframes pulse-alert {
    0%, 100% { transform: scale(1); box-shadow: 0 8px 24px rgba(255, 71, 87, 0.4); }
    50% { transform: scale(1.02); box-shadow: 0 12px 32px rgba(255, 71, 87, 0.6); }
}

.alert-icon {
    font-size: 4rem;
    animation: bounce 1s infinite;
}

@keyframes bounce {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-10px); }
}

.alert-content h2 {
    margin: 0 0 0.5rem 0;
    font-size: 1.75rem;
}

.alert-content p {
    margin: 0;
    font-size: 1.1rem;
    opacity: 0.95;
}

.waiter-action-btn {
    background: white;
    color: #FF4757;
    padding: 1rem 2rem;
    border-radius: 12px;
    font-weight: 700;
    text-decoration: none;
    transition: all 0.3s;
    white-space: nowrap;
}

.waiter-action-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.2);
}

.waiter-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.waiter-stat-card {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    box-shadow: 0 4px 16px rgba(0,0,0,0.1);
    display: flex;
    align-items: center;
    gap: 1.5rem;
    transition: all 0.3s;
    position: relative;
    overflow: hidden;
}

.waiter-stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 6px;
    height: 100%;
    transition: width 0.3s;
}

.waiter-stat-card:hover {
    transform: translateY(-6px);
    box-shadow: 0 8px 24px rgba(0,0,0,0.15);
}

.waiter-stat-card:hover::before {
    width: 100%;
    opacity: 0.1;
}

.waiter-stat-card.stat-ready::before {
    background: #FF6B35;
}

.waiter-stat-card.stat-occupied::before {
    background: #4ECDC4;
}

.waiter-stat-card.stat-today::before {
    background: #45B7D1;
}

.waiter-stat-card.stat-revenue::before {
    background: #FFA726;
}

.stat-icon-wrapper {
    width: 70px;
    height: 70px;
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

.waiter-stat-card.stat-ready .stat-icon-wrapper {
    background: linear-gradient(135deg, #FF6B35, #F7931E);
}

.waiter-stat-card.stat-occupied .stat-icon-wrapper {
    background: linear-gradient(135deg, #4ECDC4, #26A69A);
}

.waiter-stat-card.stat-today .stat-icon-wrapper {
    background: linear-gradient(135deg, #45B7D1, #2196F3);
}

.waiter-stat-card.stat-revenue .stat-icon-wrapper {
    background: linear-gradient(135deg, #FFA726, #FF9800);
}

.stat-icon {
    font-size: 2.5rem;
}

.stat-content {
    flex: 1;
}

.stat-content h3 {
    font-size: 0.875rem;
    color: #666;
    margin: 0 0 0.5rem 0;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.stat-value {
    font-size: 2rem;
    font-weight: 700;
    color: #2D3436;
    margin: 0;
    line-height: 1;
}

.stat-label {
    font-size: 0.875rem;
    color: #999;
    margin: 0.25rem 0 0 0;
}

.stat-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    background: #FF6B35;
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.875rem;
    font-weight: 600;
}

.waiter-actions-section {
    margin-bottom: 2rem;
    padding: 2rem;
    background: linear-gradient(135deg, rgba(255, 107, 53, 0.1) 0%, rgba(247, 147, 30, 0.1) 100%);
    border-radius: 20px;
    margin-top: 2rem;
}

.section-title {
    color: white;
    font-size: 1.75rem;
    margin-bottom: 1.5rem;
    text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
    font-weight: 700;
}

.waiter-quick-actions {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
}

.waiter-action-card {
    background: white;
    border-radius: 16px;
    padding: 2rem;
    text-decoration: none;
    color: inherit;
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
    transition: all 0.3s;
    box-shadow: 0 4px 16px rgba(0,0,0,0.1);
    position: relative;
    overflow: hidden;
}

.waiter-action-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    transform: scaleX(0);
    transition: transform 0.3s;
}

.waiter-action-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 12px 32px rgba(0,0,0,0.2);
}

.waiter-action-card:hover::before {
    transform: scaleX(1);
}

.action-primary::before { background: #FF6B35; }
.action-secondary::before { background: #4ECDC4; }
.action-info::before { background: #45B7D1; }
.action-warning::before { background: #FFA726; }

.action-icon {
    font-size: 3.5rem;
    margin-bottom: 1rem;
    transition: transform 0.3s;
}

.waiter-action-card:hover .action-icon {
    transform: scale(1.1) rotate(5deg);
}

.waiter-action-card h3 {
    font-size: 1.25rem;
    margin: 0 0 0.5rem 0;
    color: #2D3436;
    font-weight: 600;
}

.waiter-action-card p {
    color: #666;
    margin: 0;
    font-size: 0.9rem;
}

.action-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    background: #FF6B35;
    color: white;
    padding: 0.4rem 0.8rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
}

.waiter-recent-section {
    margin-bottom: 2rem;
}

.waiter-card {
    background: white;
    border-radius: 16px;
    box-shadow: 0 4px 16px rgba(0,0,0,0.1);
    overflow: hidden;
}

.waiter-card-header {
    background: linear-gradient(135deg, #FF6B35, #F7931E);
    padding: 1.5rem 2rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    color: white;
}

.waiter-card-header h2 {
    margin: 0;
    font-size: 1.5rem;
}

.waiter-card-body {
    padding: 1.5rem;
}

.waiter-table-responsive {
    overflow-x: auto;
}

.waiter-table {
    width: 100%;
    border-collapse: collapse;
}

.waiter-table thead {
    background: #F8F9FA;
}

.waiter-table th {
    padding: 1rem;
    text-align: left;
    font-weight: 600;
    color: #2D3436;
    border-bottom: 2px solid #E0E0E0;
}

.waiter-table td {
    padding: 1rem;
    border-bottom: 1px solid #F0F0F0;
}

.waiter-table tbody tr:hover {
    background: #FFFBF0;
}

.waiter-table-warning {
    background: #FFFBF0;
}

.waiter-badge {
    padding: 0.4rem 0.8rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 600;
    display: inline-block;
}

.waiter-badge-success {
    background: #D4EDDA;
    color: #155724;
}

.waiter-badge-warning {
    background: #FFF3CD;
    color: #856404;
}

.waiter-badge-info {
    background: #D1ECF1;
    color: #0C5460;
}

.waiter-btn-primary {
    background: #FF6B35;
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
    border: none;
    cursor: pointer;
}

.waiter-btn-primary:hover {
    background: #F7931E;
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(255, 107, 53, 0.3);
}

.waiter-btn-outline {
    background: rgba(255,255,255,0.2);
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
    border: 2px solid white;
}

.waiter-btn-outline:hover {
    background: white;
    color: #FF6B35;
}

@media (max-width: 768px) {
    .waiter-header {
        flex-direction: column;
        text-align: center;
    }
    
    .waiter-greeting h1 {
        font-size: 2rem;
    }
    
    .waiter-stats-grid,
    .waiter-quick-actions {
        grid-template-columns: 1fr;
    }
    
    .waiter-alert-card {
        flex-direction: column;
        text-align: center;
    }
}
</style>

<script>
// Update time display
function updateTime() {
    const now = new Date();
    const timeString = now.toLocaleTimeString('id-ID', { 
        hour: '2-digit', 
        minute: '2-digit',
        second: '2-digit'
    });
    const timeElement = document.getElementById('current-time');
    if (timeElement) {
        timeElement.textContent = timeString;
    }
}

updateTime();
setInterval(updateTime, 1000);
</script>

<?php else: ?>
<!-- Regular Dashboard for other roles -->
<div class="page-header">
    <h1>Dashboard</h1>
    <p>Ringkasan penjualan hari ini</p>
</div>

<div class="stats-grid">
    <div class="stat-card stat-primary">
        <div class="stat-icon">💰</div>
        <div class="stat-content">
            <h3>Total Pendapatan</h3>
            <p class="stat-value"><?php echo formatCurrency($summary['total_revenue'] ?? 0); ?></p>
        </div>
    </div>
    
    <div class="stat-card stat-success">
        <div class="stat-icon">📊</div>
        <div class="stat-content">
            <h3>Total Transaksi</h3>
            <p class="stat-value"><?php echo number_format($summary['total_transactions'] ?? 0); ?></p>
        </div>
    </div>
    
    <div class="stat-card stat-info">
        <div class="stat-icon">💵</div>
        <div class="stat-content">
            <h3>Tunai</h3>
            <p class="stat-value"><?php echo formatCurrency($summary['cash_revenue'] ?? 0); ?></p>
        </div>
    </div>
    
    <div class="stat-card stat-warning">
        <div class="stat-icon">💳</div>
        <div class="stat-content">
            <h3>Kartu</h3>
            <p class="stat-value"><?php echo formatCurrency($summary['card_revenue'] ?? 0); ?></p>
        </div>
    </div>
    
    <?php if ($unpaid_count > 0): ?>
    <div class="stat-card stat-danger">
        <div class="stat-icon">⚠️</div>
        <div class="stat-content">
            <h3>Belum Dibayar</h3>
            <p class="stat-value"><?php echo number_format($unpaid_count); ?> transaksi</p>
            <p class="stat-subvalue"><?php echo formatCurrency($unpaid_total); ?></p>
        </div>
    </div>
    <?php endif; ?>
</div>

<div class="content-grid">
    <div class="card">
        <div class="card-header">
            <h2>Transaksi Terbaru</h2>
            <a href="<?php echo BASE_URL; ?>public/transactions.php" class="btn btn-sm btn-primary">Lihat Semua</a>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Invoice</th>
                            <th>Kasir</th>
                            <th>Total</th>
                            <th>Metode</th>
                            <th>Waktu</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($transactions)): ?>
                            <tr>
                                <td colspan="6" class="text-center">Belum ada transaksi hari ini</td>
                            </tr>
                        <?php else: ?>
                        <?php foreach ($transactions as $t): 
                            $payment_status = $t['payment_status'] ?? 'paid';
                        ?>
                            <tr class="<?php echo $payment_status === 'unpaid' ? 'table-warning' : ''; ?>">
                                <td>
                                    <strong><?php echo escape($t['invoice_number']); ?></strong>
                                    <?php if ($payment_status === 'unpaid'): ?>
                                        <span class="badge badge-warning badge-sm">Belum Bayar</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo escape($t['cashier_name']); ?></td>
                                <td><?php echo formatCurrency($t['total']); ?></td>
                                <td>
                                    <span class="badge badge-info"><?php echo escape($t['payment_method']); ?></span>
                                </td>
                                <td><?php echo formatDate($t['created_at'], 'H:i'); ?></td>
                                <td>
                                    <a href="<?php echo BASE_URL; ?>public/transaction_detail.php?id=<?php echo $t['id']; ?>" class="btn btn-xs btn-primary">Detail</a>
                                    <?php if ($payment_status === 'unpaid'): ?>
                                        <a href="<?php echo BASE_URL; ?>public/payment.php?id=<?php echo $t['id']; ?>" class="btn btn-xs btn-warning">Bayar</a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2>Quick Actions</h2>
        </div>
        <div class="card-body">
            <div class="quick-actions">
                <a href="<?php echo BASE_URL; ?>public/pos.php" class="quick-action-btn">
                    <span class="icon">🛒</span>
                    <span>Buat Transaksi Baru</span>
                </a>
                <a href="<?php echo BASE_URL; ?>public/products.php" class="quick-action-btn">
                    <span class="icon">📦</span>
                    <span>Kelola Produk</span>
                </a>
                <a href="<?php echo BASE_URL; ?>public/reports.php" class="quick-action-btn">
                    <span class="icon">📈</span>
                    <span>Lihat Laporan</span>
                </a>
                <?php if ($unpaid_count > 0): ?>
                <a href="<?php echo BASE_URL; ?>public/transactions.php?filter_payment=unpaid" class="quick-action-btn" style="background: #fff3cd; border-color: #ffc107;">
                    <span class="icon">⚠️</span>
                    <span>Belum Dibayar (<?php echo $unpaid_count; ?>)</span>
                </a>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>

