<?php
/**
 * Kitchen/Order Management (for Dapur role)
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();

// Allow dapur, waiters, manager, and admin
$currentUser = getCurrentUser();
$allowed_roles = ['dapur', 'waiters', 'manager', 'admin'];
if (!in_array($currentUser['role'], $allowed_roles)) {
    setFlash('error', 'Anda tidak memiliki akses untuk halaman ini.');
    redirect(BASE_URL . 'public/dashboard.php');
}

$pageTitle = 'Dapur - Kelola Order';

$transaction = new Transaction();

// Handle status update
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'update_status') {
    $item_id = intval($_POST['item_id'] ?? 0);
    $status = $_POST['status'] ?? 'pending';
    
    if ($transaction->updateItemStatus($item_id, $status)) {
        setFlash('success', 'Status item berhasil diupdate!');
    } else {
        setFlash('error', 'Gagal mengupdate status item!');
    }
    redirect(BASE_URL . 'public/kitchen.php' . (isset($_GET['order_id']) ? '?order_id=' . $_GET['order_id'] : ''));
}

// Get selected order or list of pending orders
$selected_order_id = $_GET['order_id'] ?? null;
$date = $_GET['date'] ?? date('Y-m-d');

if ($selected_order_id) {
    $order = $transaction->getById($selected_order_id);
    $items = $transaction->getItems($selected_order_id);
} else {
    $pending_orders = $transaction->getPendingOrders($date);
}

include __DIR__ . '/includes/header.php';
?>

<style>
/* Hide navbar for kitchen view */
body .navbar {
    display: none !important;
}

body .container {
    max-width: 100% !important;
    margin: 0 !important;
    padding: 1rem !important;
}

/* Prevent zoom issues */
html, body {
    zoom: 1 !important;
    -webkit-text-size-adjust: 100% !important;
    -moz-text-size-adjust: 100% !important;
    -ms-text-size-adjust: 100% !important;
    text-size-adjust: 100% !important;
}

/* Elegant background for kitchen */
body {
    background: linear-gradient(135deg, #F0F4FF 0%, #E0E7FF 30%, #F8FAFF 60%, #EDE9FE 100%) !important;
    background-attachment: fixed;
    position: relative;
}

body::before {
    content: '';
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: 
        radial-gradient(circle at 20% 50%, rgba(99, 102, 241, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 80% 80%, rgba(139, 92, 246, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 40% 20%, rgba(236, 72, 153, 0.05) 0%, transparent 50%);
    pointer-events: none;
    z-index: 0;
}

body > * {
    position: relative;
    z-index: 1;
}
</style>

<!-- Kitchen Header - Compact -->
<div class="kitchen-header-compact">
    <div class="kitchen-title-compact">
        <h1>🏪 Dapur - Kelola Order</h1>
        <span class="kitchen-user-name"><?php echo escape($currentUser['name']); ?></span>
    </div>
    <div class="kitchen-controls-compact">
        <?php if ($selected_order_id): ?>
            <a href="<?php echo BASE_URL; ?>public/kitchen.php?date=<?php echo escape($date); ?>" class="btn-compact btn-back" title="Kembali ke Daftar Order">
                <span class="btn-icon">←</span>
                <span class="btn-label">Kembali</span>
            </a>
        <?php endif; ?>
        <a href="<?php echo BASE_URL; ?>public/dashboard.php" class="btn-compact btn-dashboard" title="Kembali ke Dashboard">
            <span class="btn-icon">🏠</span>
            <span class="btn-label">Dashboard</span>
        </a>
        <?php if ($currentUser['role'] != 'dapur'): ?>
        <a href="<?php echo BASE_URL; ?>public/pos.php" class="btn-compact btn-pos" title="Point of Sale">
            <span class="btn-icon">💰</span>
            <span class="btn-label">POS</span>
        </a>
        <?php endif; ?>
        <?php if (in_array($currentUser['role'], ['waiters', 'manager', 'admin'])): ?>
        <a href="<?php echo BASE_URL; ?>public/waiter.php" class="btn-compact btn-waiter" title="Waiter - Status Meja & Pengantaran">
            <span class="btn-icon">🍽️</span>
            <span class="btn-label">Waiter</span>
        </a>
        <?php endif; ?>
        <a href="<?php echo BASE_URL; ?>public/transactions.php" class="btn-compact btn-transactions" title="Transaksi">
            <span class="btn-icon">📋</span>
            <span class="btn-label">Transaksi</span>
        </a>
        <button class="btn-compact btn-refresh" onclick="location.reload()" title="Refresh">
            <span class="btn-icon">🔄</span>
            <span class="btn-label">Refresh</span>
        </button>
        <button class="btn-compact btn-fullscreen" id="fullscreenBtn" onclick="toggleFullscreen()" title="Fullscreen">
            <span class="btn-icon" id="fullscreenIcon">⛶</span>
            <span class="btn-label">Fullscreen</span>
        </button>
    </div>
</div>

<!-- Filter Section -->
<div class="kitchen-filter-section">
    <div class="kitchen-filter-card">
        <div class="kitchen-filter-content">
            <label for="dateFilter" class="kitchen-filter-label">📅 Filter Tanggal:</label>
            <input type="date" id="dateFilter" value="<?php echo escape($date); ?>" class="kitchen-date-input-filter">
            <button onclick="filterDate()" class="btn-kitchen-filter" title="Filter Tanggal">
                <span class="btn-icon">🔍</span>
                <span>Filter</span>
            </button>
        </div>
    </div>
</div>

<?php if ($selected_order_id && $order): ?>
    <!-- Order Detail View -->
    <div class="kitchen-card-compact">
        <div class="kitchen-card-header-compact">
            <h2>Order: <?php echo escape($order['invoice_number']); ?></h2>
            <div class="order-badges">
                <span class="badge badge-info"><?php echo formatDate($order['created_at']); ?></span>
                <span class="badge badge-primary">Kasir: <?php echo escape($order['cashier_name']); ?></span>
                <?php if (!empty($order['table_name']) || !empty($order['merged_tables'])): ?>
                    <?php if (!empty($order['is_merged']) && !empty($order['merged_tables'])): ?>
                        <span class="badge badge-warning">🪑 Gabungan: 
                            <?php 
                            $merged_names = array_map(function($t) { 
                                return $t['name']; 
                            }, $order['merged_tables']); 
                            echo escape(implode(', ', $merged_names)); 
                            ?>
                        </span>
                    <?php else: ?>
                        <span class="badge badge-success">🪑 Meja: <?php echo escape($order['table_name']); ?>
                            <?php if (!empty($order['table_location'])): ?>
                                (<?php echo escape($order['table_location']); ?>)
                            <?php endif; ?>
                        </span>
                    <?php endif; ?>
                <?php else: ?>
                    <span class="badge badge-secondary">📦 Take Away</span>
                <?php endif; ?>
            </div>
        </div>
        <div class="kitchen-card-body-compact">
            <div class="order-items-grid">
                <?php foreach ($items as $item): ?>
                    <div class="order-item-card status-<?php echo escape($item['status'] ?? 'pending'); ?>">
                        <div class="order-item-header">
                            <h3><?php echo escape($item['product_name']); ?></h3>
                            <span class="badge badge-<?php echo getStatusBadgeClass($item['status'] ?? 'pending'); ?>">
                                <?php echo getStatusLabel($item['status'] ?? 'pending'); ?>
                            </span>
                        </div>
                        <div class="order-item-details">
                            <p><strong>Kode:</strong> <?php echo escape($item['product_code']); ?></p>
                            <p><strong>Jumlah:</strong> <?php echo $item['quantity']; ?></p>
                            <p><strong>Harga:</strong> <?php echo formatCurrency($item['price']); ?> x <?php echo $item['quantity']; ?> = <?php echo formatCurrency($item['subtotal']); ?></p>
                            <?php if ($item['completed_at']): ?>
                                <p><small>Selesai: <?php echo formatDate($item['completed_at'], 'H:i'); ?></small></p>
                            <?php endif; ?>
                        </div>
                        <div class="order-item-actions">
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="update_status">
                                <input type="hidden" name="item_id" value="<?php echo $item['id']; ?>">
                                
                                <?php if (($item['status'] ?? 'pending') == 'pending'): ?>
                                    <button type="submit" name="status" value="preparing" class="btn-kitchen-action btn-kitchen-warning">Mulai Memasak</button>
                                <?php endif; ?>
                                
                                <?php if (($item['status'] ?? 'pending') == 'preparing'): ?>
                                    <button type="submit" name="status" value="ready" class="btn-kitchen-action btn-kitchen-info">Siap Disajikan</button>
                                <?php endif; ?>
                                
                                <?php if (($item['status'] ?? 'pending') == 'ready'): ?>
                                    <button type="submit" name="status" value="completed" class="btn-kitchen-action btn-kitchen-success">✓ Selesai</button>
                                <?php endif; ?>
                                
                                <?php if (($item['status'] ?? 'pending') == 'completed'): ?>
                                    <span class="text-success">✓ Item sudah selesai</span>
                                    <button type="submit" name="status" value="ready" class="btn-kitchen-action btn-kitchen-secondary">Reset ke Ready</button>
                                <?php endif; ?>
                            </form>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

<?php else: ?>
    <!-- Order List View -->
    <?php if (empty($pending_orders)): ?>
        <div class="kitchen-card-compact">
            <div class="kitchen-card-body-compact text-center">
                <p style="font-size: 1.2rem; color: #666;">Tidak ada order yang perlu diproses untuk tanggal <?php echo formatDate($date, 'd/m/Y'); ?></p>
            </div>
        </div>
    <?php else: ?>
        <div class="orders-grid">
            <?php foreach ($pending_orders as $order): ?>
                <div class="order-card">
                    <div class="order-card-header">
                        <h3><?php echo escape($order['invoice_number']); ?></h3>
                        <span class="badge badge-primary"><?php echo formatDate($order['created_at'], 'H:i'); ?></span>
                    </div>
                    <div class="order-card-body">
                        <p><strong>Kasir:</strong> <?php echo escape($order['cashier_name']); ?></p>
                        <?php if (!empty($order['table_name']) || !empty($order['merged_tables'])): ?>
                            <p><strong>🪑 Meja:</strong> 
                                <?php if (!empty($order['is_merged']) && !empty($order['merged_tables'])): ?>
                                    <?php foreach ($order['merged_tables'] as $merged_table): ?>
                                        <span class="badge badge-warning" style="margin-right: 5px;">
                                            <?php echo escape($merged_table['name']); ?>
                                            <?php if (!empty($merged_table['location'])): ?>
                                                (<?php echo escape($merged_table['location']); ?>)
                                            <?php endif; ?>
                                        </span>
                                    <?php endforeach; ?>
                                    <small class="text-muted">(Gabungan)</small>
                                <?php else: ?>
                                    <span class="badge badge-info">
                                        <?php echo escape($order['table_name']); ?>
                                        <?php if (!empty($order['table_location'])): ?>
                                            (<?php echo escape($order['table_location']); ?>)
                                        <?php endif; ?>
                                    </span>
                                <?php endif; ?>
                            </p>
                        <?php else: ?>
                            <p><strong>📦 Take Away</strong></p>
                        <?php endif; ?>
                        <div class="order-stats">
                            <div class="stat-item">
                                <span class="stat-label">Total Item:</span>
                                <span class="stat-value"><?php echo $order['total_items']; ?></span>
                            </div>
                            <?php if ($order['pending_count'] > 0): ?>
                                <div class="stat-item stat-pending">
                                    <span class="stat-label">Pending:</span>
                                    <span class="stat-value"><?php echo $order['pending_count']; ?></span>
                                </div>
                            <?php endif; ?>
                            <?php if ($order['preparing_count'] > 0): ?>
                                <div class="stat-item stat-preparing">
                                    <span class="stat-label">Memasak:</span>
                                    <span class="stat-value"><?php echo $order['preparing_count']; ?></span>
                                </div>
                            <?php endif; ?>
                            <?php if ($order['ready_count'] > 0): ?>
                                <div class="stat-item stat-ready">
                                    <span class="stat-label">Siap:</span>
                                    <span class="stat-value"><?php echo $order['ready_count']; ?></span>
                                </div>
                            <?php endif; ?>
                            <?php if ($order['completed_count'] > 0): ?>
                                <div class="stat-item stat-completed">
                                    <span class="stat-label">Selesai:</span>
                                    <span class="stat-value"><?php echo $order['completed_count']; ?></span>
                                </div>
                            <?php endif; ?>
                        </div>
                        <a href="<?php echo BASE_URL; ?>public/kitchen.php?order_id=<?php echo $order['id']; ?>&date=<?php echo escape($date); ?>" class="btn-kitchen-primary btn-block">Lihat Detail & Kelola</a>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
<?php endif; ?>

<style>
/* Kitchen Header - Compact */
.kitchen-header-compact {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 30%, #7C3AED 60%, #EC4899 100%);
    padding: 0.875rem 1.25rem;
    border-radius: 16px;
    margin-bottom: 1rem;
    box-shadow: 0 6px 24px rgba(99, 102, 241, 0.35), 0 0 0 1px rgba(255, 255, 255, 0.1) inset;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: sticky;
    top: 0;
    z-index: 100;
    backdrop-filter: blur(10px);
    position: relative;
    overflow: hidden;
}

.kitchen-header-compact::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    animation: headerShine 3s infinite;
}

@keyframes headerShine {
    0% {
        left: -100%;
    }
    100% {
        left: 100%;
    }
}

.kitchen-title-compact {
    position: relative;
    z-index: 1;
}

.kitchen-title-compact h1 {
    color: white;
    font-size: 1.5rem;
    margin: 0;
    font-weight: 800;
    display: inline-block;
    text-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
    letter-spacing: -0.5px;
}

.kitchen-user-name {
    color: rgba(255,255,255,0.9);
    font-size: 0.875rem;
    margin-left: 0.75rem;
    font-weight: 500;
}

.kitchen-controls-compact {
    display: flex;
    gap: 0.5rem;
    align-items: center;
}

/* Kitchen Filter Section */
.kitchen-filter-section {
    margin-bottom: 1.5rem;
}

.kitchen-filter-card {
    background: linear-gradient(135deg, #FFFFFF 0%, #F8FAFF 100%);
    border-radius: 16px;
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.1), 0 0 0 1px rgba(224, 231, 255, 0.5) inset;
    border: 1px solid rgba(224, 231, 255, 0.8);
    padding: 1rem 1.5rem;
    position: relative;
    overflow: hidden;
}

.kitchen-filter-card::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    opacity: 0.3;
}

.kitchen-filter-content {
    display: flex;
    align-items: center;
    gap: 1rem;
    flex-wrap: wrap;
}

.kitchen-filter-label {
    font-weight: 600;
    font-size: 0.9375rem;
    color: #4B5563;
    white-space: nowrap;
}

.kitchen-date-input-filter {
    padding: 0.625rem 0.875rem;
    border: 2px solid #E0E7FF;
    border-radius: 8px;
    background: white;
    color: #1F2937;
    font-size: 0.875rem;
    min-height: 44px;
    transition: all 0.2s;
    flex: 1;
    min-width: 180px;
}

.kitchen-date-input-filter:focus {
    outline: none;
    border-color: #6366F1;
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.kitchen-date-input-filter::-webkit-calendar-picker-indicator {
    cursor: pointer;
    filter: opacity(0.7);
}

.kitchen-date-input-filter::-webkit-calendar-picker-indicator:hover {
    filter: opacity(1);
}

.btn-kitchen-filter {
    background: linear-gradient(135deg, #6366F1 0%, #8B5CF6 50%, #EC4899 100%);
    color: white;
    border: none;
    padding: 0.625rem 1.25rem;
    border-radius: 8px;
    font-weight: 600;
    font-size: 0.875rem;
    cursor: pointer;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.35);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    min-height: 44px;
    white-space: nowrap;
}

.btn-kitchen-filter:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(99, 102, 241, 0.5);
}

.btn-kitchen-filter:active {
    transform: translateY(0) scale(0.98);
}

.btn-kitchen-filter .btn-icon {
    font-size: 1rem;
}

.btn-compact {
    background: rgba(255,255,255,0.2);
    color: white;
    border: 2px solid rgba(255,255,255,0.5);
    border-radius: 8px;
    padding: 0.5rem 0.625rem;
    font-size: 1.25rem;
    cursor: pointer;
    transition: all 0.2s;
    min-width: 60px;
    min-height: 60px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 0.25rem;
    text-decoration: none;
}

.btn-compact:hover {
    background: white;
    color: #6366F1;
    transform: scale(1.05);
    text-decoration: none;
}

.btn-icon {
    font-size: 1.25rem;
    line-height: 1;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-label {
    font-size: 0.625rem;
    font-weight: 600;
    line-height: 1;
    text-align: center;
    white-space: nowrap;
    letter-spacing: 0.02em;
}

/* Kitchen Cards */
.kitchen-card-compact {
    background: linear-gradient(135deg, #FFFFFF 0%, #F8FAFF 100%);
    border-radius: 16px;
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.1), 0 0 0 1px rgba(224, 231, 255, 0.5) inset;
    overflow: hidden;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid rgba(224, 231, 255, 0.8);
    position: relative;
}

.kitchen-card-compact::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.kitchen-card-compact:hover {
    box-shadow: 0 8px 32px rgba(99, 102, 241, 0.2), 0 0 0 1px rgba(99, 102, 241, 0.3) inset;
    transform: translateY(-2px);
}

.kitchen-card-compact:hover::after {
    opacity: 1;
}

.kitchen-card-header-compact {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 50%, #7C3AED 100%);
    color: white;
    padding: 0.875rem 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 0.5rem;
    position: relative;
    overflow: hidden;
}

.kitchen-card-header-compact::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: shimmer 3s infinite;
}

@keyframes shimmer {
    0%, 100% {
        transform: translate(0, 0) rotate(0deg);
    }
    50% {
        transform: translate(50%, 50%) rotate(180deg);
    }
}

.kitchen-card-header-compact h2 {
    margin: 0;
    font-size: 1rem;
    font-weight: 600;
}

.order-badges {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
}

.kitchen-card-body-compact {
    padding: 1rem;
}

/* Orders Grid */
.orders-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 1.5rem;
}

.order-card {
    background: linear-gradient(135deg, #FFFFFF 0%, #F8FAFF 100%);
    border-radius: 16px;
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.1), 0 0 0 1px rgba(224, 231, 255, 0.5) inset;
    overflow: hidden;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid rgba(224, 231, 255, 0.8);
    position: relative;
}

.order-card::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.order-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 32px rgba(99, 102, 241, 0.2), 0 0 0 1px rgba(99, 102, 241, 0.3) inset;
}

.order-card:hover::after {
    opacity: 1;
}

.order-card-header {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 50%, #7C3AED 100%);
    color: white;
    padding: 1rem 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    overflow: hidden;
}

.order-card-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: shimmer 3s infinite;
}

.order-card-header h3 {
    margin: 0;
    font-size: 1.125rem;
    font-weight: 700;
}

.order-card-body {
    padding: 1.5rem;
}

.order-stats {
    display: flex;
    flex-wrap: wrap;
    gap: 1rem;
    margin: 1rem 0;
}

.stat-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    padding: 0.5rem;
    border-radius: 8px;
    background: #f8f9fa;
    min-width: 60px;
    transition: all 0.2s;
}

.stat-item:hover {
    transform: scale(1.05);
}

.stat-item.stat-pending { 
    background: linear-gradient(135deg, #FEF3C7, #FDE68A);
    border: 1px solid #FCD34D;
}

.stat-item.stat-preparing { 
    background: linear-gradient(135deg, #DBEAFE, #BFDBFE);
    border: 1px solid #93C5FD;
}

.stat-item.stat-ready { 
    background: linear-gradient(135deg, #D1FAE5, #A7F3D0);
    border: 1px solid #6EE7B7;
}

.stat-item.stat-completed { 
    background: linear-gradient(135deg, #D1FAE5, #A7F3D0);
    border: 1px solid #6EE7B7;
}

.stat-label {
    font-size: 0.75rem;
    color: #666;
    margin-bottom: 0.25rem;
    font-weight: 600;
}

.stat-value {
    font-size: 1.5rem;
    font-weight: bold;
    color: #333;
}

/* Order Items Grid */
.order-items-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 1.5rem;
}

.order-item-card {
    background: linear-gradient(135deg, #FFFFFF 0%, #F8FAFF 100%);
    border: 2px solid #e0e0e0;
    border-radius: 12px;
    padding: 1.5rem;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.order-item-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 3px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    transform: scaleX(0);
    transform-origin: left;
    transition: transform 0.3s ease;
}

.order-item-card:hover::before {
    transform: scaleX(1);
}

.order-item-card.status-pending {
    border-color: #ffc107;
    background: linear-gradient(135deg, #FFFBF0 0%, #FFFFFF 100%);
}

.order-item-card.status-preparing {
    border-color: #17a2b8;
    background: linear-gradient(135deg, #F0F9FF 0%, #FFFFFF 100%);
}

.order-item-card.status-ready {
    border-color: #28a745;
    background: linear-gradient(135deg, #F0FFF4 0%, #FFFFFF 100%);
}

.order-item-card.status-completed {
    border-color: #28a745;
    background: linear-gradient(135deg, #F0FFF4 0%, #FFFFFF 100%);
    opacity: 0.8;
}

.order-item-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 2px solid #e0e0e0;
}

.order-item-header h3 {
    margin: 0;
    font-size: 1.125rem;
    color: #333;
    font-weight: 700;
}

.order-item-details {
    margin-bottom: 1rem;
}

.order-item-details p {
    margin: 0.5rem 0;
    font-size: 0.875rem;
    color: #4B5563;
}

.order-item-actions {
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 1px solid #e0e0e0;
}

/* Kitchen Action Buttons */
.btn-kitchen-action {
    padding: 0.625rem 1rem;
    border-radius: 8px;
    font-weight: 600;
    font-size: 0.875rem;
    border: none;
    cursor: pointer;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.btn-kitchen-action:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.btn-kitchen-action:active {
    transform: translateY(0) scale(0.98);
}

.btn-kitchen-warning {
    background: linear-gradient(135deg, #F59E0B, #D97706);
    color: white;
}

.btn-kitchen-info {
    background: linear-gradient(135deg, #3B82F6, #2563EB);
    color: white;
}

.btn-kitchen-success {
    background: linear-gradient(135deg, #10B981, #059669);
    color: white;
}

.btn-kitchen-secondary {
    background: linear-gradient(135deg, #6B7280, #4B5563);
    color: white;
}

.btn-kitchen-primary {
    background: linear-gradient(135deg, #6366F1 0%, #8B5CF6 50%, #EC4899 100%);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 10px;
    font-weight: 700;
    font-size: 0.9375rem;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    width: 100%;
    text-align: center;
    text-decoration: none;
    display: block;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.35);
}

.btn-kitchen-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(99, 102, 241, 0.5);
    text-decoration: none;
    color: white;
}

.btn-kitchen-primary:active {
    transform: translateY(0) scale(0.98);
}

/* Fullscreen Mode */
body.kitchen-fullscreen-active .kitchen-header-compact {
    margin-bottom: 0.75rem;
    padding: 0.625rem 0.875rem;
}

body.kitchen-fullscreen-active .container {
    max-width: 100%;
    margin: 0;
    padding: 0.5rem;
}

@media (max-width: 768px) {
    .orders-grid, .order-items-grid {
        grid-template-columns: 1fr;
    }
    
    .kitchen-header-compact {
        flex-direction: column;
        gap: 0.5rem;
        text-align: center;
    }
    
    .kitchen-controls-compact {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .kitchen-filter-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .kitchen-date-input-filter {
        width: 100%;
        min-width: 100%;
    }
    
    .btn-kitchen-filter {
        width: 100%;
        justify-content: center;
    }
}
</style>

<script>
function filterDate() {
    const date = document.getElementById('dateFilter').value;
    window.location.href = '<?php echo BASE_URL; ?>public/kitchen.php?date=' + date;
}

// Enter fullscreen function
function enterFullscreen() {
    const elem = document.documentElement;
    
    if (!document.fullscreenElement && !document.webkitFullscreenElement && 
        !document.mozFullScreenElement && !document.msFullscreenElement) {
        // Try to enter fullscreen
        const promise = elem.requestFullscreen ? elem.requestFullscreen() :
                       elem.webkitRequestFullscreen ? elem.webkitRequestFullscreen() :
                       elem.mozRequestFullScreen ? elem.mozRequestFullScreen() :
                       elem.msRequestFullscreen ? elem.msRequestFullscreen() : null;
        
        if (promise) {
            promise.then(() => {
                // Fullscreen successful, update button state
                updateFullscreenButton();
            }).catch(err => {
                // Fullscreen failed (may require user interaction in some browsers)
                console.log('Fullscreen request failed:', err);
            });
        }
    }
}

// Exit fullscreen function
function exitFullscreen() {
    if (document.exitFullscreen) {
        document.exitFullscreen();
    } else if (document.webkitExitFullscreen) {
        document.webkitExitFullscreen();
    } else if (document.mozCancelFullScreen) {
        document.mozCancelFullScreen();
    } else if (document.msExitFullscreen) {
        document.msExitFullscreen();
    }
}

// Fullscreen functionality
function toggleFullscreen() {
    const elem = document.documentElement;
    const fullscreenBtn = document.getElementById('fullscreenBtn');
    const fullscreenIcon = document.getElementById('fullscreenIcon');
    
    if (!document.fullscreenElement && !document.webkitFullscreenElement && 
        !document.mozFullScreenElement && !document.msFullscreenElement) {
        // Enter fullscreen
        enterFullscreen();
        
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        document.body.classList.add('kitchen-fullscreen-active');
    } else {
        // Exit fullscreen
        exitFullscreen();
        
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        document.body.classList.remove('kitchen-fullscreen-active');
    }
}

// Listen for fullscreen changes
document.addEventListener('fullscreenchange', updateFullscreenButton);
document.addEventListener('webkitfullscreenchange', updateFullscreenButton);
document.addEventListener('mozfullscreenchange', updateFullscreenButton);
document.addEventListener('MSFullscreenChange', updateFullscreenButton);

function updateFullscreenButton() {
    const isFullscreen = !!(document.fullscreenElement || document.webkitFullscreenElement || 
                            document.mozFullScreenElement || document.msFullscreenElement);
    const fullscreenIcon = document.getElementById('fullscreenIcon');
    
    if (isFullscreen) {
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        document.body.classList.add('kitchen-fullscreen-active');
    } else {
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        document.body.classList.remove('kitchen-fullscreen-active');
    }
}

// Keyboard shortcut for fullscreen (F11)
document.addEventListener('keydown', function(e) {
    if (e.key === 'F11') {
        e.preventDefault();
        toggleFullscreen();
    }
});

// Auto refresh every 30 seconds
setInterval(function() {
    if (!document.hidden && <?php echo $selected_order_id ? 'false' : 'true'; ?>) {
        location.reload();
    }
}, 30000);
</script>

<?php 
function getStatusLabel($status) {
    $labels = [
        'pending' => 'Menunggu',
        'preparing' => 'Memasak',
        'ready' => 'Siap',
        'completed' => 'Selesai'
    ];
    return $labels[$status] ?? 'Unknown';
}

function getStatusBadgeClass($status) {
    $classes = [
        'pending' => 'warning',
        'preparing' => 'info',
        'ready' => 'success',
        'completed' => 'success'
    ];
    return $classes[$status] ?? 'secondary';
}
?>

<?php include __DIR__ . '/includes/footer.php'; ?>

