<?php
/**
 * Database Migration Runner
 * CLI and Web Interface
 * 
 * Usage:
 *   CLI: php migrate.php [status|run|rollback]
 *   Web: Access via browser
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/classes/Migrator.php';

$migrator = new Migrator();

// Check if running from CLI
$isCli = php_sapi_name() === 'cli';

if ($isCli) {
    // CLI Mode
    $action = $argv[1] ?? 'status';
    
    switch ($action) {
        case 'status':
            $status = $migrator->getStatus();
            echo "=== Migration Status ===\n";
            echo "Total migrations: {$status['total']}\n";
            echo "Applied: {$status['applied']}\n";
            echo "Pending: {$status['pending']}\n";
            
            if (!empty($status['pending_files'])) {
                echo "\nPending migrations:\n";
                foreach ($status['pending_files'] as $file) {
                    echo "  - {$file}\n";
                }
            }
            break;
            
        case 'run':
            echo "Running pending migrations...\n\n";
            $result = $migrator->runPendingMigrations();
            
            if ($result['success']) {
                echo "✅ {$result['message']}\n\n";
            } else {
                echo "❌ Error: {$result['message']}\n\n";
            }
            
            if (!empty($result['results'])) {
                foreach ($result['results'] as $r) {
                    $file = $r['file'];
                    $res = $r['result'];
                    if ($res['success']) {
                        $time = isset($res['execution_time']) ? " ({$res['execution_time']}s)" : "";
                        echo "✅ {$file}{$time}\n";
                    } else {
                        echo "❌ {$file}: {$res['message']}\n";
                    }
                }
            }
            break;
            
        case 'rollback':
            echo "Rolling back last batch...\n\n";
            $result = $migrator->rollbackLastBatch();
            
            if ($result['success']) {
                echo "✅ {$result['message']}\n\n";
            } else {
                echo "❌ {$result['message']}\n\n";
            }
            
            if (!empty($result['results'])) {
                foreach ($result['results'] as $r) {
                    if ($r['success']) {
                        echo "✅ {$r['migration']}: {$r['message']}\n";
                    } else {
                        echo "❌ {$r['migration']}: {$r['message']}\n";
                    }
                }
            }
            break;
            
        default:
            echo "Usage: php migrate.php [status|run|rollback]\n";
            break;
    }
} else {
    // Web Mode
    require_once __DIR__ . '/../config/functions.php';
    requireLogin();
    
    // Only allow admin and manager
    $currentUser = getCurrentUser();
    if (!in_array($currentUser['role'], ['admin', 'manager'])) {
        setFlash('error', 'Anda tidak memiliki akses untuk halaman ini.');
        redirect(BASE_URL . 'public/dashboard.php');
    }
    
    $pageTitle = 'Database Migrations';
    
    $action = $_GET['action'] ?? 'status';
    $message = '';
    $messageType = '';
    $results = null;
    
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        $action = $_POST['action'] ?? 'status';
        
        switch ($action) {
            case 'run':
                $result = $migrator->runPendingMigrations();
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'error';
                $results = $result['results'] ?? [];
                break;
                
            case 'rollback':
                $result = $migrator->rollbackLastBatch();
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'error';
                $results = $result['results'] ?? [];
                break;
        }
    }
    
    $status = $migrator->getStatus();
    
    include __DIR__ . '/includes/header.php';
    ?>
    
    <div class="page-header">
        <h1>Database Migrations</h1>
        <p>Kelola update database secara otomatis</p>
    </div>
    
    <?php if ($message): ?>
        <div class="alert alert-<?php echo $messageType; ?>">
            <?php echo escape($message); ?>
        </div>
    <?php endif; ?>
    
    <div class="card">
        <div class="card-header">
            <h2>Status Migrasi</h2>
        </div>
        <div class="card-body">
            <div class="stats-grid" style="grid-template-columns: repeat(3, 1fr);">
                <div class="stat-card stat-primary">
                    <div class="stat-content">
                        <h3>Total Migrations</h3>
                        <p class="stat-value"><?php echo $status['total']; ?></p>
                    </div>
                </div>
                <div class="stat-card stat-success">
                    <div class="stat-content">
                        <h3>Applied</h3>
                        <p class="stat-value"><?php echo $status['applied']; ?></p>
                    </div>
                </div>
                <div class="stat-card stat-warning">
                    <div class="stat-content">
                        <h3>Pending</h3>
                        <p class="stat-value"><?php echo $status['pending']; ?></p>
                    </div>
                </div>
            </div>
            
            <?php if (!empty($status['pending_files'])): ?>
                <div style="margin-top: 2rem;">
                    <h3>Pending Migrations:</h3>
                    <ul>
                        <?php foreach ($status['pending_files'] as $file): ?>
                            <li><code><?php echo escape($file); ?></code></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php else: ?>
                <div class="alert alert-success" style="margin-top: 2rem;">
                    ✅ Semua migrasi sudah diterapkan!
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <?php if (!empty($status['pending_files'])): ?>
    <div class="card">
        <div class="card-header">
            <h2>Actions</h2>
        </div>
        <div class="card-body">
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="run">
                <button type="submit" class="btn btn-success" onclick="return confirm('Yakin ingin menjalankan semua pending migrations?')">
                    ▶️ Run Pending Migrations
                </button>
            </form>
            
            <form method="POST" style="display: inline; margin-left: 1rem;">
                <input type="hidden" name="action" value="rollback">
                <button type="submit" class="btn btn-warning" onclick="return confirm('Yakin ingin rollback batch terakhir? Ini akan membatalkan perubahan!')">
                    ⏪ Rollback Last Batch
                </button>
            </form>
        </div>
    </div>
    <?php endif; ?>
    
    <?php if ($results !== null && !empty($results)): ?>
    <div class="card">
        <div class="card-header">
            <h2>Results</h2>
        </div>
        <div class="card-body">
            <table class="table">
                <thead>
                    <tr>
                        <th>Migration</th>
                        <th>Status</th>
                        <th>Message</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($results as $r): 
                        $migrationName = $r['migration'] ?? $r['file'] ?? 'Unknown';
                        $result = $r['result'] ?? $r;
                        $success = $result['success'] ?? false;
                    ?>
                        <tr>
                            <td><code><?php echo escape($migrationName); ?></code></td>
                            <td>
                                <?php if ($success): ?>
                                    <span class="badge badge-success">Success</span>
                                <?php else: ?>
                                    <span class="badge badge-danger">Failed</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo escape($result['message'] ?? ''); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
    
    <div class="card">
        <div class="card-header">
            <h2>Info</h2>
        </div>
        <div class="card-body">
            <p><strong>Migrations Path:</strong> <code><?php echo escape(__DIR__ . '/../migrations'); ?></code></p>
            <p><strong>CLI Usage:</strong></p>
            <pre style="background: #f5f5f5; padding: 1rem; border-radius: 4px;">
php migrate.php status    # Check status
php migrate.php run        # Run pending migrations
php migrate.php rollback   # Rollback last batch
            </pre>
        </div>
    </div>
    
    <?php include __DIR__ . '/includes/footer.php'; ?>
    <?php
}
?>

