<?php
/**
 * Payment Page - Complete payment for unpaid transactions
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();

$pageTitle = 'Pembayaran';

$transaction = new Transaction();
$id = $_GET['id'] ?? 0;

// Get transaction
$trans = $transaction->getById($id);
if (!$trans) {
    setFlash('error', 'Transaksi tidak ditemukan!');
    redirect(BASE_URL . 'public/transactions.php');
}

// Check if already paid
if ($trans['payment_status'] === 'paid') {
    setFlash('info', 'Transaksi ini sudah dibayar!');
    redirect(BASE_URL . 'public/transaction_detail.php?id=' . $id);
}

// Handle payment
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'complete_payment') {
    $payment_method = $_POST['payment_method'] ?? 'cash';
    $cash_received = floatval($_POST['cash_received'] ?? $trans['total']);
    
    // Validation
    if ($payment_method === 'cash' && $cash_received < $trans['total']) {
        setFlash('error', 'Uang yang diterima kurang dari total pembayaran!');
        redirect(BASE_URL . 'public/payment.php?id=' . $id);
    }
    
    if ($transaction->completePayment($id, $payment_method, $cash_received)) {
        setFlash('success', 'Pembayaran berhasil diselesaikan!');
        redirect(BASE_URL . 'public/transaction_detail.php?id=' . $id);
    } else {
        setFlash('error', 'Gagal menyelesaikan pembayaran!');
        redirect(BASE_URL . 'public/payment.php?id=' . $id);
    }
}

$items = $transaction->getItems($id);

include __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1>Pembayaran Transaksi</h1>
    <a href="<?php echo BASE_URL; ?>public/transaction_detail.php?id=<?php echo $id; ?>" class="btn btn-secondary">Kembali ke Detail</a>
</div>

<div class="content-grid">
    <div class="card">
        <div class="card-header">
            <h2>Detail Transaksi</h2>
        </div>
        <div class="card-body">
            <div class="transaction-info">
                <p><strong>Invoice:</strong> <?php echo escape($trans['invoice_number']); ?></p>
                <p><strong>Tanggal Order:</strong> <?php echo formatDate($trans['created_at']); ?></p>
                <p><strong>Kasir:</strong> <?php echo escape($trans['cashier_name']); ?></p>
                <?php if (!empty($trans['table_name']) || !empty($trans['merged_tables'])): ?>
                    <?php if (!empty($trans['is_merged']) && !empty($trans['merged_tables'])): ?>
                        <p><strong>🪑 Meja Gabungan:</strong> 
                            <?php foreach ($trans['merged_tables'] as $merged_table): ?>
                                <span style="display: inline-block; margin-right: 5px;">
                                    <strong><?php echo escape($merged_table['name']); ?></strong>
                                    <?php if (!empty($merged_table['location'])): ?>
                                        (<?php echo escape($merged_table['location']); ?>)
                                    <?php endif; ?>
                                </span>
                            <?php endforeach; ?>
                        </p>
                    <?php else: ?>
                        <p><strong>🪑 Meja:</strong> <?php echo escape($trans['table_name']); ?>
                            <?php if (!empty($trans['table_location'])): ?>
                                (<?php echo escape($trans['table_location']); ?>)
                            <?php endif; ?>
                        </p>
                    <?php endif; ?>
                <?php else: ?>
                    <p><strong>📦 Tipe:</strong> Take Away</p>
                <?php endif; ?>
            </div>
            
            <hr>
            
            <table class="table">
                <thead>
                    <tr>
                        <th>Item</th>
                        <th class="text-right">Qty</th>
                        <th class="text-right">Harga</th>
                        <th class="text-right">Subtotal</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $item): ?>
                        <tr>
                            <td>
                                <?php echo escape($item['product_name']); ?><br>
                                <small class="text-muted"><?php echo escape($item['product_code']); ?></small>
                            </td>
                            <td class="text-right"><?php echo $item['quantity']; ?></td>
                            <td class="text-right"><?php echo formatCurrency($item['price']); ?></td>
                            <td class="text-right"><?php echo formatCurrency($item['subtotal']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
                <tfoot>
                    <tr class="total-row">
                        <td colspan="3" class="text-right"><strong>Total:</strong></td>
                        <td class="text-right"><strong><?php echo formatCurrency($trans['total']); ?></strong></td>
                    </tr>
                </tfoot>
            </table>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header">
            <h2>Pembayaran</h2>
        </div>
        <div class="card-body">
            <form method="POST" id="paymentForm">
                <input type="hidden" name="action" value="complete_payment">
                
                <div class="alert alert-warning">
                    <strong>⚠️ Status:</strong> Transaksi ini belum dibayar. Silakan selesaikan pembayaran.
                </div>
                
                <div class="form-group">
                    <label>Metode Pembayaran</label>
                    <select name="payment_method" id="paymentMethod" class="form-control" required>
                        <option value="cash">Tunai</option>
                        <option value="card">Kartu</option>
                        <option value="qris">QRIS</option>
                    </select>
                </div>
                
                <div class="form-group" id="cashReceivedGroup">
                    <label>Uang Diterima</label>
                    <input type="number" name="cash_received" id="cashReceived" class="form-control" 
                           min="<?php echo $trans['total']; ?>" step="1000" 
                           value="<?php echo $trans['total']; ?>" required>
                    <small class="text-muted" id="changeAmount"></small>
                </div>
                
                <div class="payment-summary">
                    <div class="summary-row">
                        <span>Total Tagihan:</span>
                        <span><strong><?php echo formatCurrency($trans['total']); ?></strong></span>
                    </div>
                    <div class="summary-row" id="changeRow" style="display:none;">
                        <span>Kembalian:</span>
                        <span id="changeValue"><strong>Rp 0</strong></span>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-success btn-lg btn-block">
                    Selesaikan Pembayaran
                </button>
            </form>
        </div>
    </div>
</div>

<script>
const total = <?php echo $trans['total']; ?>;
const cashReceivedInput = document.getElementById('cashReceived');
const changeAmount = document.getElementById('changeAmount');
const changeRow = document.getElementById('changeRow');
const changeValue = document.getElementById('changeValue');
const paymentMethod = document.getElementById('paymentMethod');

function formatCurrency(amount) {
    return 'Rp ' + parseInt(amount).toLocaleString('id-ID');
}

function updateChange() {
    const cashReceived = parseFloat(cashReceivedInput.value || 0);
    
    if (paymentMethod.value === 'cash') {
        if (cashReceived >= total) {
            const change = cashReceived - total;
            changeAmount.textContent = 'Kembalian: ' + formatCurrency(change);
            changeAmount.style.color = '#28a745';
            changeRow.style.display = 'flex';
            changeValue.innerHTML = '<strong>' + formatCurrency(change) + '</strong>';
        } else {
            changeAmount.textContent = 'Uang kurang: ' + formatCurrency(total - cashReceived);
            changeAmount.style.color = '#dc3545';
            changeRow.style.display = 'none';
        }
    } else {
        changeAmount.textContent = '';
        changeRow.style.display = 'none';
    }
}

paymentMethod.addEventListener('change', function() {
    const cashReceivedGroup = document.getElementById('cashReceivedGroup');
    if (this.value === 'cash') {
        cashReceivedGroup.style.display = 'block';
        cashReceivedInput.value = total;
        updateChange();
    } else {
        cashReceivedGroup.style.display = 'none';
        cashReceivedInput.value = total;
        changeAmount.textContent = '';
        changeRow.style.display = 'none';
    }
});

cashReceivedInput.addEventListener('input', updateChange);

// Initialize
updateChange();

// Form validation
document.getElementById('paymentForm').addEventListener('submit', function(e) {
    if (paymentMethod.value === 'cash') {
        const cashReceived = parseFloat(cashReceivedInput.value || 0);
        if (cashReceived < total) {
            e.preventDefault();
            alert('Uang yang diterima kurang dari total pembayaran!');
            return false;
        }
    }
});
</script>

<style>
.transaction-info p {
    margin: 0.5rem 0;
}

.payment-summary {
    background: #f8f9fa;
    padding: 1rem;
    border-radius: 4px;
    margin: 1rem 0;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    padding: 0.5rem 0;
    border-bottom: 1px solid #dee2e6;
}

.summary-row:last-child {
    border-bottom: none;
}

.total-row {
    background: #f8f9fa;
    font-size: 1.1rem;
}
</style>

<?php include __DIR__ . '/includes/footer.php'; ?>






