<?php
/**
 * Point of Sale Interface
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();

$currentUser = getCurrentUser();
$isWaiter = ($currentUser['role'] == 'waiters');

$pageTitle = 'Point of Sale';

$product = new Product();
$category = new Category();
$transaction = new Transaction();
$table = new Table();

// Handle transaction
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'checkout') {
    $items = json_decode($_POST['items'], true);
    $total = floatval($_POST['total'] ?? 0);
    $payment_method = $_POST['payment_method'] ?? 'cash';
    $cash_received = floatval($_POST['cash_received'] ?? $total);
    $pay_later = isset($_POST['pay_later']) && $_POST['pay_later'] === '1';
    
    // Handle multiple table selection
    $table_ids = null;
    if (!empty($_POST['table_ids']) && is_array($_POST['table_ids'])) {
        $table_ids = array_map('intval', $_POST['table_ids']);
        $table_ids = array_filter($table_ids); // Remove empty values
        $table_ids = !empty($table_ids) ? array_values($table_ids) : null;
    } elseif (!empty($_POST['table_id'])) {
        // Backward compatibility: single table_id
        $table_ids = [intval($_POST['table_id'])];
    }
    
    // Quick validation before processing
    if (empty($items) || !is_array($items) || count($items) === 0) {
        setFlash('error', 'Keranjang kosong!');
        redirect(BASE_URL . 'public/pos.php');
    }
    
    if ($total <= 0) {
        setFlash('error', 'Total pembayaran tidak valid!');
        redirect(BASE_URL . 'public/pos.php');
    }
    
    // Validate payment only if not paying later
    if (!$pay_later && $payment_method === 'cash' && $cash_received < $total) {
        setFlash('error', 'Uang yang diterima kurang dari total pembayaran!');
        redirect(BASE_URL . 'public/pos.php');
    }
    
    // Determine payment status
    $payment_status = $pay_later ? 'unpaid' : 'paid';
    
    $transaction_id = $transaction->create($items, $total, $payment_method, $cash_received, $table_ids, $payment_status);
    
    if ($transaction_id) {
        if ($pay_later) {
            setFlash('success', 'Transaksi berhasil dibuat! Pelanggan dapat membayar nanti setelah selesai makan.');
        }
        redirect(BASE_URL . 'public/transaction_detail.php?id=' . $transaction_id);
    } else {
        setFlash('error', 'Gagal memproses transaksi!');
        redirect(BASE_URL . 'public/pos.php');
    }
}

$categories = $category->getAll();
$products = $product->getAll();
$available_tables = $table->getAvailable();

// Pass data to JavaScript for offline storage
$products_json = json_encode($products);
$categories_json = json_encode($categories);

include __DIR__ . '/includes/header.php';
?>
<?php if ($isWaiter): ?>
<style>
/* Hide navbar for waiters in POS */
body .navbar {
    display: none !important;
}

body .container {
    max-width: 100% !important;
    margin: 0 !important;
    padding: 1rem !important;
}

/* Prevent zoom issues */
html, body {
    zoom: 1 !important;
    -webkit-text-size-adjust: 100% !important;
    -moz-text-size-adjust: 100% !important;
    -ms-text-size-adjust: 100% !important;
    text-size-adjust: 100% !important;
}

/* Elegant background for waiter POS */
body {
    background: linear-gradient(135deg, #F0F4FF 0%, #E0E7FF 30%, #F8FAFF 60%, #EDE9FE 100%) !important;
    background-attachment: fixed;
    position: relative;
}

body::before {
    content: '';
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: 
        radial-gradient(circle at 20% 50%, rgba(99, 102, 241, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 80% 80%, rgba(139, 92, 246, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 40% 20%, rgba(236, 72, 153, 0.05) 0%, transparent 50%);
    pointer-events: none;
    z-index: 0;
}

body > * {
    position: relative;
    z-index: 1;
}
</style>
<?php endif; ?>

<?php if ($isWaiter): ?>
<!-- Waiter POS Header - Compact -->
<div class="waiter-pos-header-compact">
    <div class="waiter-pos-title-compact">
        <h1>🍽️ POS</h1>
        <span class="waiter-name"><?php echo escape($currentUser['name']); ?></span>
    </div>
    <div class="waiter-pos-controls-compact">
        <a href="<?php echo BASE_URL; ?>public/dashboard.php" class="btn-compact btn-dashboard" title="Kembali ke Dashboard">
            <span class="btn-icon">🏠</span>
            <span class="btn-label">Dashboard</span>
        </a>
        <a href="<?php echo BASE_URL; ?>public/waiter.php" class="btn-compact btn-waiter" title="Waiter - Status Meja & Pengantaran">
            <span class="btn-icon">🍽️</span>
            <span class="btn-label">Waiter</span>
        </a>
        <?php if (in_array($currentUser['role'], ['dapur', 'waiters', 'manager', 'admin'])): ?>
        <a href="<?php echo BASE_URL; ?>public/kitchen.php" class="btn-compact btn-kitchen" title="Dapur - Status Pesanan">
            <span class="btn-icon">🏪</span>
            <span class="btn-label">Dapur</span>
        </a>
        <?php endif; ?>
        <a href="<?php echo BASE_URL; ?>public/transactions.php" class="btn-compact btn-transactions" title="Transaksi">
            <span class="btn-icon">📋</span>
            <span class="btn-label">Transaksi</span>
        </a>
        <button class="btn-compact btn-refresh" onclick="location.reload()" title="Refresh">
            <span class="btn-icon">🔄</span>
            <span class="btn-label">Refresh</span>
        </button>
        <button class="btn-compact btn-fullscreen" id="fullscreenBtn" onclick="toggleFullscreen()" title="Fullscreen">
            <span class="btn-icon" id="fullscreenIcon">⛶</span>
            <span class="btn-label">Fullscreen</span>
        </button>
    </div>
</div>
<?php else: ?>
<div class="page-header">
    <h1>Point of Sale</h1>
</div>
<?php endif; ?>

<div class="pos-container <?php echo $isWaiter ? 'waiter-pos-container-compact' : ''; ?>">
    <div class="pos-left <?php echo $isWaiter ? 'waiter-pos-left-compact' : ''; ?>">
        <?php if ($isWaiter): ?>
        <!-- Sticky Category Bar for Waiter -->
        <div class="waiter-category-sticky">
            <div class="waiter-search-compact">
                <input type="text" id="searchProduct" placeholder="🔍 Cari..." class="waiter-search-input-compact">
            </div>
            <div class="category-tabs-compact">
                <button class="category-tab-compact active" data-category="all">Semua</button>
                <?php foreach ($categories as $cat): ?>
                    <button class="category-tab-compact" data-category="<?php echo $cat['id']; ?>">
                        <?php echo escape($cat['name']); ?>
                    </button>
                <?php endforeach; ?>
            </div>
        </div>
        <?php else: ?>
        <div class="card">
            <div class="card-header">
                <h2>📂 Kategori</h2>
            </div>
            <div class="card-body">
                <div class="category-tabs">
                    <button class="category-tab active" data-category="all">Semua</button>
                    <?php foreach ($categories as $cat): ?>
                        <button class="category-tab" data-category="<?php echo $cat['id']; ?>">
                            <?php echo escape($cat['name']); ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="card <?php echo $isWaiter ? 'waiter-card-compact' : ''; ?>">
            <?php if (!$isWaiter): ?>
            <div class="card-header">
                <h2>🛍️ Produk</h2>
                <input type="text" id="searchProduct" placeholder="🔍 Cari produk..." class="form-control" style="width:200px;">
            </div>
            <?php endif; ?>
            <div class="card-body <?php echo $isWaiter ? 'waiter-card-body-compact' : ''; ?>">
                <div class="products-grid <?php echo $isWaiter ? 'waiter-products-grid-compact' : ''; ?>" id="productsGrid">
                    <?php foreach ($products as $prod): 
                        $category_name = '';
                        foreach ($categories as $cat) {
                            if ($cat['id'] == $prod['category_id']) {
                                $category_name = $cat['name'];
                                break;
                            }
                        }
                        $stock_percentage = $prod['stock'] > 0 ? min(100, ($prod['stock'] / 50) * 100) : 0;
                    ?>
                        <div class="product-item <?php echo $isWaiter ? 'waiter-product-item-compact' : ''; ?>" data-category="<?php echo $prod['category_id']; ?>" data-name="<?php echo strtolower($prod['name']); ?>">
                            <div class="product-info">
                                <div class="product-header-info">
                                    <h3><?php echo escape($prod['name']); ?></h3>
                                    <?php if ($category_name): ?>
                                        <span class="product-category-badge"><?php echo escape($category_name); ?></span>
                                    <?php endif; ?>
                                </div>
                                <p class="product-code"><?php echo escape($prod['code']); ?></p>
                                <div class="product-price-section">
                                    <p class="product-price"><?php echo formatCurrency($prod['price']); ?></p>
                                </div>
                                <div class="product-stock-info">
                                    <?php if ($prod['stock'] <= 0): ?>
                                        <div class="stock-indicator stock-empty">
                                            <span class="stock-badge">
                                                <span class="stock-icon">✕</span>
                                                <span class="stock-label">Habis</span>
                                            </span>
                                        </div>
                                    <?php elseif ($prod['stock'] <= 10): ?>
                                        <div class="stock-indicator stock-low">
                                            <span class="stock-badge">
                                                <span class="stock-icon">⚠</span>
                                                <span class="stock-label"><?php echo $prod['stock']; ?> unit</span>
                                            </span>
                                            <div class="stock-bar">
                                                <div class="stock-fill" style="width: <?php echo $stock_percentage; ?>%"></div>
                                            </div>
                                        </div>
                                    <?php else: ?>
                                        <div class="stock-indicator stock-ok">
                                            <span class="stock-badge stock-badge-success">
                                                <span class="stock-icon">✓</span>
                                                <span class="stock-label">Tersedia</span>
                                            </span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <button class="btn <?php echo $isWaiter ? 'btn-waiter-add' : 'btn-sm btn-primary'; ?> btn-block" 
                                    onclick="addToCart(<?php echo htmlspecialchars(json_encode($prod)); ?>)"
                                    <?php echo $prod['stock'] <= 0 ? 'disabled' : ''; ?>>
                                <span class="btn-icon">+</span>
                                <span class="btn-text">Tambah</span>
                            </button>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
    
    <div class="pos-right <?php echo $isWaiter ? 'waiter-pos-right-compact' : ''; ?>">
        <div class="card <?php echo $isWaiter ? 'waiter-card-compact waiter-cart-sticky' : ''; ?>">
            <div class="card-header <?php echo $isWaiter ? 'waiter-card-header-compact' : ''; ?>">
                <h2>🛒 Keranjang</h2>
                <button class="btn <?php echo $isWaiter ? 'btn-waiter-clear' : 'btn-sm btn-secondary'; ?>" onclick="clearCart()">🗑️</button>
            </div>
            <div class="card-body <?php echo $isWaiter ? 'waiter-card-body-compact' : ''; ?>">
                <div id="cartItems" class="<?php echo $isWaiter ? 'waiter-cart-items' : ''; ?>">
                    <p class="text-center text-muted">Keranjang kosong</p>
                </div>
                <div class="cart-summary <?php echo $isWaiter ? 'waiter-cart-summary-compact' : ''; ?>" id="cartSummary" style="display:none;">
                    <div class="summary-row">
                        <span>Subtotal:</span>
                        <span id="subtotal">Rp 0</span>
                    </div>
                    <div class="summary-row total">
                        <span><strong>Total:</strong></span>
                        <span id="total"><strong>Rp 0</strong></span>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="card <?php echo $isWaiter ? 'waiter-card-compact' : ''; ?>">
            <div class="card-header <?php echo $isWaiter ? 'waiter-card-header-compact' : ''; ?>">
                <h2>💳 Pembayaran</h2>
            </div>
            <div class="card-body <?php echo $isWaiter ? 'waiter-card-body-compact' : ''; ?>">
                <form id="checkoutForm" method="POST">
                    <input type="hidden" name="action" value="checkout">
                    <input type="hidden" name="items" id="cartItemsInput">
                    <input type="hidden" name="total" id="cartTotalInput">
                    
                    <div class="payment-form-compact">
                        <div class="form-group-compact">
                            <label class="form-label-compact">🪑 Meja</label>
                            <div id="tableSelection" class="table-selection-compact">
                                <label class="table-option-compact">
                                    <input type="checkbox" id="noTableCheckbox" onchange="toggleTableSelection()">
                                    <span>Take Away</span>
                                </label>
                                <?php foreach ($available_tables as $tbl): ?>
                                    <label class="table-option-compact">
                                        <input type="checkbox" name="table_ids[]" value="<?php echo $tbl['id']; ?>" 
                                               class="table-checkbox" onchange="updateTableSelection()">
                                        <span><?php echo escape($tbl['name']); ?> <small>(<?php echo $tbl['capacity']; ?>)</small></span>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                            <small class="form-info-compact" id="tableSelectionInfo">Pilih meja atau Take Away</small>
                        </div>
                        
                        <div class="form-group-compact">
                            <label class="checkbox-compact">
                                <input type="checkbox" name="pay_later" id="payLater" value="1" onchange="togglePayLater()">
                                <span>💳 Bayar Nanti</span>
                            </label>
                        </div>
                        
                        <div id="paymentSection" class="payment-method-section">
                            <div class="form-row-compact">
                                <div class="form-group-compact">
                                    <label class="form-label-compact">Metode</label>
                                    <select name="payment_method" id="paymentMethod" class="form-input-compact">
                                        <option value="cash">💵 Tunai</option>
                                        <option value="card">💳 Kartu</option>
                                        <option value="qris">📱 QRIS</option>
                                    </select>
                                </div>
                                
                                <div class="form-group-compact" id="cashReceivedGroup">
                                    <label class="form-label-compact">Uang Diterima</label>
                                    <input type="number" name="cash_received" id="cashReceived" class="form-input-compact" min="0" step="1000" value="0">
                                    <small class="form-info-compact" id="changeAmount"></small>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-checkout-compact" id="checkoutBtn" disabled>
                        <span id="checkoutBtnText">✅ Proses</span>
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Save products and categories snapshot for offline use
(async function() {
    try {
        await offlineDB.init();
        
        // Save products snapshot
        const products = <?php echo json_encode($products); ?>;
        if (products && products.length > 0) {
            await offlineDB.saveProductsSnapshot(products);
            console.log('[POS] Products snapshot saved for offline use');
        }
        
        // Save categories snapshot
        const categories = <?php echo json_encode($categories); ?>;
        if (categories && categories.length > 0) {
            await offlineDB.saveCategoriesSnapshot(categories);
            console.log('[POS] Categories snapshot saved for offline use');
        }
    } catch (error) {
        console.error('[POS] Error saving snapshots:', error);
    }
})();

// Prevent zoom issues - Reset zoom immediately
(function() {
    if (document.body) {
        document.body.style.zoom = '1';
    }
    if (document.documentElement) {
        document.documentElement.style.zoom = '1';
    }
    
    // Also set on window load
    window.addEventListener('load', function() {
        if (document.body.style.zoom !== '1') {
            document.body.style.zoom = '1';
        }
        if (document.documentElement.style.zoom !== '1') {
            document.documentElement.style.zoom = '1';
        }
    });
})();

let cart = [];

function addToCart(product) {
    const existingIndex = cart.findIndex(item => item.product_id === product.id);
    
    if (existingIndex !== -1) {
        const existingItem = cart[existingIndex];
        if (existingItem.quantity >= product.stock) {
            alert('Stok tidak mencukupi!');
            return;
        }
        existingItem.quantity++;
        existingItem.subtotal = existingItem.quantity * existingItem.price;
        // Move to top when quantity is updated
        cart.splice(existingIndex, 1);
        cart.unshift(existingItem);
    } else {
        if (product.stock <= 0) {
            alert('Produk habis!');
            return;
        }
        // Add new item at the top
        cart.unshift({
            product_id: product.id,
            product_name: product.name,
            product_code: product.code,
            price: parseFloat(product.price),
            quantity: 1,
            subtotal: parseFloat(product.price)
        });
    }
    
    updateCart();
}

function removeFromCart(index) {
    cart.splice(index, 1);
    updateCart();
}

function updateQuantity(index, change) {
    const item = cart[index];
    const newQuantity = item.quantity + change;
    
    if (newQuantity <= 0) {
        removeFromCart(index);
        return;
    }
    
    // Check stock (you might want to fetch current stock from server)
    item.quantity = newQuantity;
    item.subtotal = item.quantity * item.price;
    
    // Move to top when quantity is updated
    if (index > 0) {
        cart.splice(index, 1);
        cart.unshift(item);
    }
    
    updateCart();
}

function clearCart() {
    if (confirm('Yakin ingin mengosongkan keranjang?')) {
        cart = [];
        updateCart();
    }
}

function updateCart() {
    const cartItemsDiv = document.getElementById('cartItems');
    const cartSummary = document.getElementById('cartSummary');
    const checkoutBtn = document.getElementById('checkoutBtn');
    
    if (cart.length === 0) {
        cartItemsDiv.innerHTML = '<p class="text-center text-muted">Keranjang kosong</p>';
        cartSummary.style.display = 'none';
        checkoutBtn.disabled = true;
        return;
    }
    
    const isWaiter = <?php echo $isWaiter ? 'true' : 'false'; ?>;
    let html = '<div class="cart-items-list">';
    let subtotal = 0;
    
    cart.forEach((item, index) => {
        subtotal += item.subtotal;
        const btnClass = isWaiter ? 'btn-waiter-cart' : 'btn-xs';
        const dangerClass = isWaiter ? 'btn-waiter-danger' : 'btn-danger';
        html += `
            <div class="cart-item-compact" data-index="${index}">
                <div class="cart-item-main">
                    <div class="cart-item-title">
                        <span class="cart-item-name">${item.product_name}</span>
                        <span class="cart-item-code">${item.product_code}</span>
                    </div>
                    <div class="cart-item-price">${formatCurrency(item.subtotal)}</div>
                </div>
                <div class="cart-item-actions">
                    <div class="quantity-control-compact">
                        <button class="qty-btn qty-minus" onclick="updateQuantity(${index}, -1)">−</button>
                        <span class="qty-value">${item.quantity}</span>
                        <button class="qty-btn qty-plus" onclick="updateQuantity(${index}, 1)">+</button>
                    </div>
                    <button class="btn-remove-compact" onclick="removeFromCart(${index})" title="Hapus">×</button>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    cartItemsDiv.innerHTML = html;
    cartSummary.style.display = 'block';
    
    document.getElementById('subtotal').textContent = formatCurrency(subtotal);
    document.getElementById('total').innerHTML = '<strong>' + formatCurrency(subtotal) + '</strong>';
    
    document.getElementById('cartItemsInput').value = JSON.stringify(cart);
    document.getElementById('cartTotalInput').value = subtotal;
    
    const cashReceived = document.getElementById('cashReceived');
    if (!cashReceived.value || parseFloat(cashReceived.value) < subtotal) {
        cashReceived.value = subtotal;
    }
    updateChange();
    
    checkoutBtn.disabled = false;
}

function formatCurrency(amount) {
    return 'Rp ' + parseInt(amount).toLocaleString('id-ID');
}

function updateChange() {
    const total = parseFloat(document.getElementById('cartTotalInput').value || 0);
    const cashReceived = parseFloat(document.getElementById('cashReceived').value || 0);
    const changeAmount = document.getElementById('changeAmount');
    
    if (cashReceived >= total) {
        const change = cashReceived - total;
        changeAmount.textContent = 'Kembalian: ' + formatCurrency(change);
        changeAmount.style.color = '#28a745';
    } else {
        changeAmount.textContent = 'Uang kurang: ' + formatCurrency(total - cashReceived);
        changeAmount.style.color = '#dc3545';
    }
}

// Category filter - Handle both regular and compact tabs
function setupCategoryFilter() {
    const allTabs = document.querySelectorAll('.category-tab, .category-tab-compact');
    
    allTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            // Remove active from all tabs in the same container
            const container = this.closest('.category-tabs, .category-tabs-compact');
            if (container) {
                container.querySelectorAll('.category-tab, .category-tab-compact').forEach(t => {
                    t.classList.remove('active');
                });
            }
            
            // Add active to clicked tab
            this.classList.add('active');
            
            // Get category ID
            const categoryId = this.getAttribute('data-category');
            const searchInput = document.getElementById('searchProduct');
            const searchTerm = searchInput ? searchInput.value.toLowerCase() : '';
            
            // Filter products
            document.querySelectorAll('.product-item').forEach(item => {
                const itemCategory = item.getAttribute('data-category');
                const itemName = item.getAttribute('data-name') || '';
                
                // Check category filter
                const categoryMatch = categoryId === 'all' || itemCategory == categoryId;
                
                // Check search filter
                const searchMatch = !searchTerm || itemName.includes(searchTerm);
                
                // Show if both match
                if (categoryMatch && searchMatch) {
                    item.style.display = 'block';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    });
}

// Initialize category filter
setupCategoryFilter();

// Search filter - Combined with category filter
const searchInput = document.getElementById('searchProduct');
if (searchInput) {
    searchInput.addEventListener('input', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        
        // Get active category
        const activeTab = document.querySelector('.category-tab.active, .category-tab-compact.active');
        const categoryId = activeTab ? activeTab.getAttribute('data-category') : 'all';
        
        // Filter products
        document.querySelectorAll('.product-item').forEach(item => {
            const itemCategory = item.getAttribute('data-category');
            const itemName = item.getAttribute('data-name') || '';
            
            // Check category filter
            const categoryMatch = categoryId === 'all' || itemCategory == categoryId;
            
            // Check search filter
            const searchMatch = !searchTerm || itemName.includes(searchTerm);
            
            // Show if both match
            if (categoryMatch && searchMatch) {
                item.style.display = 'block';
            } else {
                item.style.display = 'none';
            }
        });
    });
}

// Toggle pay later
function togglePayLater() {
    const payLater = document.getElementById('payLater').checked;
    const paymentSection = document.getElementById('paymentSection');
    const checkoutBtn = document.getElementById('checkoutBtn');
    const checkoutBtnText = document.getElementById('checkoutBtnText');
    
    if (payLater) {
        paymentSection.style.display = 'none';
        checkoutBtnText.textContent = 'Buat Transaksi (Bayar Nanti)';
        checkoutBtn.classList.remove('btn-success');
        checkoutBtn.classList.add('btn-warning');
    } else {
        paymentSection.style.display = 'block';
        checkoutBtnText.textContent = 'Proses Transaksi';
        checkoutBtn.classList.remove('btn-warning');
        checkoutBtn.classList.add('btn-success');
        updateChange();
    }
}

// Payment method change
document.getElementById('paymentMethod').addEventListener('change', function() {
    const cashReceivedGroup = document.getElementById('cashReceivedGroup');
    if (this.value === 'cash') {
        cashReceivedGroup.style.display = 'block';
    } else {
        cashReceivedGroup.style.display = 'none';
        document.getElementById('cashReceived').value = document.getElementById('cartTotalInput').value;
    }
});

// Cash received change
document.getElementById('cashReceived').addEventListener('input', updateChange);

// Table selection functions
function toggleTableSelection() {
    const noTableCheckbox = document.getElementById('noTableCheckbox');
    const tableCheckboxes = document.querySelectorAll('.table-checkbox');
    
    if (noTableCheckbox.checked) {
        // Uncheck all table checkboxes
        tableCheckboxes.forEach(cb => cb.checked = false);
    }
    updateTableSelection();
}

function updateTableSelection() {
    const noTableCheckbox = document.getElementById('noTableCheckbox');
    const tableCheckboxes = document.querySelectorAll('.table-checkbox:checked');
    const infoElement = document.getElementById('tableSelectionInfo');
    
    if (noTableCheckbox.checked) {
        infoElement.textContent = 'Take Away - Tidak ada meja yang dipilih';
        infoElement.style.color = '#666';
    } else if (tableCheckboxes.length === 0) {
        infoElement.textContent = 'Pilih satu atau lebih meja untuk dine-in, atau centang "Tanpa Meja" untuk take away';
        infoElement.style.color = '#666';
    } else if (tableCheckboxes.length === 1) {
        const tableName = tableCheckboxes[0].closest('label').textContent.trim();
        infoElement.textContent = `Meja dipilih: ${tableName}`;
        infoElement.style.color = '#28a745';
    } else {
        const tableNames = Array.from(tableCheckboxes).map(cb => {
            return cb.closest('label').textContent.trim().split('(')[0].trim();
        });
        infoElement.textContent = `Meja digabung: ${tableNames.join(', ')} (${tableCheckboxes.length} meja)`;
        infoElement.style.color = '#ffc107';
        infoElement.style.fontWeight = 'bold';
    }
}

// Checkout form - Support offline mode
document.getElementById('checkoutForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    if (cart.length === 0) {
        alert('Keranjang kosong!');
        return false;
    }
    
    const payLater = document.getElementById('payLater').checked;
    const paymentMethod = document.getElementById('paymentMethod').value;
    const total = parseFloat(document.getElementById('cartTotalInput').value);
    const cashReceived = parseFloat(document.getElementById('cashReceived').value);
    
    // Only validate payment if not paying later
    if (!payLater && paymentMethod === 'cash' && cashReceived < total) {
        alert('Uang yang diterima kurang dari total pembayaran!');
        return false;
    }
    
    // Get table selection
    const noTableCheckbox = document.getElementById('noTableCheckbox');
    const tableCheckboxes = document.querySelectorAll('.table-checkbox:checked');
    const tableIds = [];
    
    if (!noTableCheckbox.checked && tableCheckboxes.length > 0) {
        tableCheckboxes.forEach(cb => {
            tableIds.push(parseInt(cb.value));
        });
    }
    
    // Check if online or offline
    const isOnline = navigator.onLine;
    
    if (isOnline) {
        // Online mode - submit normally
        // Temporarily disable offline handling to use normal form submission
        const form = document.getElementById('checkoutForm');
        const originalSubmit = form.onsubmit;
        form.onsubmit = null;
        form.submit();
    } else {
        // Offline mode - save to IndexedDB
        try {
            // Show loading
            const checkoutBtn = document.getElementById('checkoutBtn');
            const originalText = checkoutBtn.innerHTML;
            checkoutBtn.disabled = true;
            checkoutBtn.innerHTML = 'Menyimpan...';
            
            // Prepare transaction data
            const transactionData = {
                items: cart.map(item => ({
                    product_id: item.product_id,
                    product_name: item.product_name,
                    product_code: item.product_code,
                    quantity: item.quantity,
                    price: item.price,
                    subtotal: item.subtotal
                })),
                total: total,
                payment_method: paymentMethod,
                cash_received: cashReceived,
                table_ids: tableIds.length > 0 ? tableIds : null,
                payment_status: payLater ? 'unpaid' : 'paid',
                user_id: <?php echo getUserId(); ?>,
                created_at: new Date().toISOString()
            };
            
            // Save to IndexedDB
            await offlineDB.init();
            const transactionId = await offlineDB.saveTransaction(transactionData);
            
            // Clear cart
            cart = [];
            updateCart();
            
            // Reset form
            document.getElementById('checkoutForm').reset();
            document.getElementById('noTableCheckbox').checked = false;
            
            // Show success message
            alert('Transaksi berhasil disimpan secara offline!\nTransaksi akan disinkronkan otomatis saat koneksi internet kembali.');
            
            // Update status indicator
            if (statusIndicator) {
                await statusIndicator.updatePendingCount();
                statusIndicator.updateIndicator();
            }
            
            // Reset button
            checkoutBtn.disabled = false;
            checkoutBtn.innerHTML = originalText;
            
        } catch (error) {
            console.error('Error saving offline transaction:', error);
            alert('Gagal menyimpan transaksi offline. Silakan coba lagi atau hubungi administrator.');
            
            const checkoutBtn = document.getElementById('checkoutBtn');
            checkoutBtn.disabled = false;
            checkoutBtn.innerHTML = '✅ Proses';
        }
    }
});

// Enter fullscreen function
function enterFullscreen() {
    const elem = document.documentElement;
    
    if (!document.fullscreenElement && !document.webkitFullscreenElement && 
        !document.mozFullScreenElement && !document.msFullscreenElement) {
        // Try to enter fullscreen
        const promise = elem.requestFullscreen ? elem.requestFullscreen() :
                       elem.webkitRequestFullscreen ? elem.webkitRequestFullscreen() :
                       elem.mozRequestFullScreen ? elem.mozRequestFullScreen() :
                       elem.msRequestFullscreen ? elem.msRequestFullscreen() : null;
        
        if (promise) {
            promise.then(() => {
                // Fullscreen successful, update button state
                updateFullscreenButton();
            }).catch(err => {
                // Fullscreen failed (may require user interaction in some browsers)
                console.log('Fullscreen request failed:', err);
            });
        }
    }
}

// Exit fullscreen function
function exitFullscreen() {
    if (document.exitFullscreen) {
        document.exitFullscreen();
    } else if (document.webkitExitFullscreen) {
        document.webkitExitFullscreen();
    } else if (document.mozCancelFullScreen) {
        document.mozCancelFullScreen();
    } else if (document.msExitFullscreen) {
        document.msExitFullscreen();
    }
}

// Fullscreen functionality
function toggleFullscreen() {
    const elem = document.documentElement;
    const fullscreenBtn = document.getElementById('fullscreenBtn');
    const fullscreenIcon = document.getElementById('fullscreenIcon');
    const fullscreenText = document.getElementById('fullscreenText');
    
    if (!document.fullscreenElement && !document.webkitFullscreenElement && 
        !document.mozFullScreenElement && !document.msFullscreenElement) {
        // Enter fullscreen
        enterFullscreen();
        
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Exit Fullscreen';
        document.body.classList.add('waiter-fullscreen-active');
    } else {
        // Exit fullscreen
        exitFullscreen();
        
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Fullscreen';
        document.body.classList.remove('waiter-fullscreen-active');
    }
}

// Listen for fullscreen changes
document.addEventListener('fullscreenchange', updateFullscreenButton);
document.addEventListener('webkitfullscreenchange', updateFullscreenButton);
document.addEventListener('mozfullscreenchange', updateFullscreenButton);
document.addEventListener('MSFullscreenChange', updateFullscreenButton);

function updateFullscreenButton() {
    const isFullscreen = !!(document.fullscreenElement || document.webkitFullscreenElement || 
                            document.mozFullScreenElement || document.msFullscreenElement);
    const fullscreenIcon = document.getElementById('fullscreenIcon');
    const fullscreenText = document.getElementById('fullscreenText');
    
    if (isFullscreen) {
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Exit Fullscreen';
        document.body.classList.add('waiter-fullscreen-active');
    } else {
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Fullscreen';
        document.body.classList.remove('waiter-fullscreen-active');
    }
}

// Keyboard shortcut for fullscreen (F11) - only for waiters
<?php if ($isWaiter): ?>
document.addEventListener('keydown', function(e) {
    if (e.key === 'F11') {
        e.preventDefault();
        toggleFullscreen();
    }
});

// Initialize on page load - for waiters
document.addEventListener('DOMContentLoaded', function() {
    // Reset zoom level to 100%
    if (document.body.style.zoom) {
        document.body.style.zoom = '1';
    }
    if (document.documentElement.style.zoom) {
        document.documentElement.style.zoom = '1';
    }
    
    // Ensure all products are visible initially
    document.querySelectorAll('.product-item').forEach(item => {
        item.style.display = 'block';
    });
    
    // Ensure "Semua" tab is active for waiters
    const allTab = document.querySelector('.category-tab-compact[data-category="all"]');
    if (allTab) {
        document.querySelectorAll('.category-tab-compact').forEach(t => t.classList.remove('active'));
        allTab.classList.add('active');
    }
});
<?php endif; ?>


<?php if ($isWaiter): ?>
// Prevent zoom level changes - for waiters
window.addEventListener('resize', function() {
    // Reset zoom if it gets changed
    if (document.body.style.zoom && document.body.style.zoom !== '1') {
        document.body.style.zoom = '1';
    }
    if (document.documentElement.style.zoom && document.documentElement.style.zoom !== '1') {
        document.documentElement.style.zoom = '1';
    }
});

// Reset zoom on fullscreen change - for waiters
document.addEventListener('fullscreenchange', function() {
    setTimeout(function() {
        if (document.body.style.zoom && document.body.style.zoom !== '1') {
            document.body.style.zoom = '1';
        }
        if (document.documentElement.style.zoom && document.documentElement.style.zoom !== '1') {
            document.documentElement.style.zoom = '1';
        }
    }, 100);
});

document.addEventListener('webkitfullscreenchange', function() {
    setTimeout(function() {
        if (document.body.style.zoom && document.body.style.zoom !== '1') {
            document.body.style.zoom = '1';
        }
        if (document.documentElement.style.zoom && document.documentElement.style.zoom !== '1') {
            document.documentElement.style.zoom = '1';
        }
    }, 100);
});
<?php endif; ?>
</script>

<style>
/* General Cart Item Styles - Compact & Clean */
.cart-items-list {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

/* Limit cart items display to 2 items with scroll */
#cartItems:not(.waiter-cart-items) {
    max-height: 220px;
    overflow-y: auto;
    -webkit-overflow-scrolling: touch;
}

.cart-item-compact {
    background: white;
    border: 1.5px solid #E0E7FF;
    border-radius: 8px;
    padding: 0.5rem 0.625rem;
    transition: all 0.2s ease;
    animation: slideInCart 0.3s ease;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

@keyframes slideInCart {
    from {
        opacity: 0;
        transform: translateY(-8px) scale(0.98);
    }
    to {
        opacity: 1;
        transform: translateY(0) scale(1);
    }
}

.cart-item-compact:hover {
    border-color: #6366F1;
    box-shadow: 0 2px 8px rgba(99, 102, 241, 0.15);
    background: #F8FAFF;
}

.cart-item-main {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 0.5rem;
}

.cart-item-title {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.125rem;
    min-width: 0;
}

.cart-item-name {
    font-size: 0.8125rem;
    color: #1F2937;
    font-weight: 600;
    line-height: 1.3;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.cart-item-code {
    font-size: 0.6875rem;
    color: #9CA3AF;
    font-family: 'Courier New', monospace;
    font-weight: 400;
}

.cart-item-price {
    font-size: 0.875rem;
    color: #6366F1;
    font-weight: 700;
    white-space: nowrap;
    flex-shrink: 0;
}

.cart-item-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 0.5rem;
    padding-top: 0.375rem;
    border-top: 1px solid #E5E7EB;
}

.quantity-control-compact {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    background: #F8FAFF;
    padding: 0.125rem;
    border-radius: 6px;
    border: 1px solid #E0E7FF;
}

.qty-btn {
    background: linear-gradient(135deg, #6366F1, #4F46E5);
    color: white;
    border: none;
    width: 26px;
    height: 26px;
    border-radius: 4px;
    cursor: pointer;
    font-weight: 700;
    font-size: 0.875rem;
    transition: all 0.15s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    line-height: 1;
    padding: 0;
}

.qty-btn:hover {
    background: linear-gradient(135deg, #8B5CF6, #6366F1);
    transform: scale(1.05);
}

.qty-btn:active {
    transform: scale(0.95);
}

.qty-value {
    min-width: 24px;
    text-align: center;
    font-weight: 700;
    color: #1F2937;
    font-size: 0.8125rem;
    padding: 0 0.25rem;
}

.btn-remove-compact {
    background: #EF4444;
    color: white;
    border: none;
    width: 26px;
    height: 26px;
    border-radius: 4px;
    cursor: pointer;
    font-weight: 700;
    font-size: 1rem;
    transition: all 0.15s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    line-height: 1;
    padding: 0;
    flex-shrink: 0;
}

.btn-remove-compact:hover {
    background: #DC2626;
    transform: scale(1.05);
}

.btn-remove-compact:active {
    transform: scale(0.95);
}

/* Legacy support */
.cart-item-card {
    background: white;
    border: 1.5px solid #E0E7FF;
    border-radius: 8px;
    padding: 0.5rem 0.625rem;
}

.cart-item-info {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.cart-item-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 0.5rem;
}

.cart-item-name-section {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.125rem;
}

.btn-remove {
    flex-shrink: 0;
    width: 26px;
    height: 26px;
    padding: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1rem;
    line-height: 1;
}

.cart-item-controls {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 0.5rem;
    padding-top: 0.375rem;
    border-top: 1px solid #E5E7EB;
}

.quantity-control {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    background: #F8FAFF;
    padding: 0.125rem;
    border-radius: 6px;
    border: 1px solid #E0E7FF;
}

.quantity-control button {
    background: linear-gradient(135deg, #6366F1, #4F46E5);
    color: white;
    border: none;
    width: 26px;
    height: 26px;
    border-radius: 4px;
    cursor: pointer;
    font-weight: 700;
    font-size: 0.875rem;
    transition: all 0.15s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    line-height: 1;
    padding: 0;
}

.quantity-control button:hover {
    background: linear-gradient(135deg, #8B5CF6, #6366F1);
    transform: scale(1.05);
}

.quantity-control button:active {
    transform: scale(0.95);
}

.quantity {
    min-width: 24px;
    text-align: center;
    font-weight: 700;
    color: #1F2937;
    font-size: 0.8125rem;
    padding: 0 0.25rem;
}

.cart-item-subtotal {
    display: flex;
    align-items: center;
}

.subtotal-amount {
    font-size: 0.875rem;
    color: #6366F1;
    font-weight: 700;
}
</style>

<?php if ($isWaiter): ?>
<style>
/* ============================================
   COMPACT & TABLET-FRIENDLY WAITER POS STYLES
   ============================================ */

/* Compact Header */
.waiter-pos-header-compact {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 30%, #7C3AED 60%, #EC4899 100%);
    padding: 0.875rem 1.25rem;
    border-radius: 16px;
    margin-bottom: 1rem;
    box-shadow: 0 6px 24px rgba(99, 102, 241, 0.35), 0 0 0 1px rgba(255, 255, 255, 0.1) inset;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: sticky;
    top: 0;
    z-index: 100;
    backdrop-filter: blur(10px);
    position: relative;
    overflow: hidden;
}

.waiter-pos-header-compact::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    animation: headerShine 3s infinite;
}

@keyframes headerShine {
    0% {
        left: -100%;
    }
    100% {
        left: 100%;
    }
}

.waiter-pos-title-compact {
    position: relative;
    z-index: 1;
}

.waiter-pos-title-compact h1 {
    color: white;
    font-size: 1.5rem;
    margin: 0;
    font-weight: 800;
    display: inline-block;
    text-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
    letter-spacing: -0.5px;
}

.waiter-name {
    color: rgba(255,255,255,0.9);
    font-size: 0.875rem;
    margin-left: 0.75rem;
    font-weight: 500;
}

.waiter-pos-controls-compact {
    display: flex;
    gap: 0.5rem;
}

.btn-compact {
    background: rgba(255,255,255,0.2);
    color: white;
    border: 2px solid rgba(255,255,255,0.5);
    border-radius: 8px;
    padding: 0.5rem 0.625rem;
    font-size: 1.25rem;
    cursor: pointer;
    transition: all 0.2s;
    min-width: 60px;
    min-height: 60px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 0.25rem;
    text-decoration: none;
}

.btn-compact:hover {
    background: white;
    color: #6366F1;
    transform: scale(1.05);
    text-decoration: none;
}

.btn-icon {
    font-size: 1.25rem;
    line-height: 1;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-label {
    font-size: 0.625rem;
    font-weight: 600;
    line-height: 1;
    text-align: center;
    white-space: nowrap;
    letter-spacing: 0.02em;
}

/* Container */
.waiter-pos-container-compact {
    gap: 1rem;
    display: grid;
    grid-template-columns: 2fr 1fr;
}

.waiter-pos-left-compact {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.waiter-pos-right-compact {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

/* Sticky Category Bar */
.waiter-category-sticky {
    background: white;
    padding: 0.75rem 1rem;
    border-radius: 12px;
    box-shadow: 0 2px 12px rgba(99, 102, 241, 0.1);
    position: sticky;
    top: 0;
    z-index: 90;
    margin-bottom: 0;
    border: 1px solid #E0E7FF;
}

.waiter-search-compact {
    margin-bottom: 0.75rem;
}

.waiter-search-input-compact {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #C7D2FE;
    border-radius: 8px;
    font-size: 1rem;
    transition: all 0.2s;
    background: #F8FAFF;
}

.waiter-search-input-compact:focus {
    outline: none;
    border-color: #6366F1;
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
    background: white;
}

.category-tabs-compact {
    display: flex;
    gap: 0.5rem;
    flex-wrap: nowrap;
    overflow-x: auto;
    overflow-y: hidden;
    -webkit-overflow-scrolling: touch;
    padding: 0.5rem 0.25rem;
    scrollbar-width: thin;
    scrollbar-color: #C7D2FE #F8FAFF;
    margin: 0 -0.25rem;
}

.category-tabs-compact::-webkit-scrollbar {
    height: 6px;
}

.category-tabs-compact::-webkit-scrollbar-track {
    background: #F8FAFF;
    border-radius: 3px;
}

.category-tabs-compact::-webkit-scrollbar-thumb {
    background: #C7D2FE;
    border-radius: 3px;
}

.category-tabs-compact::-webkit-scrollbar-thumb:hover {
    background: #6366F1;
}

.category-tab-compact {
    padding: 0.625rem 1.25rem;
    background: #F0F4FF;
    border: 2px solid #C7D2FE;
    border-radius: 8px;
    cursor: pointer;
    font-weight: 600;
    font-size: 0.875rem;
    transition: all 0.2s ease;
    color: #4B5563;
    white-space: nowrap;
    min-height: 44px;
    min-width: fit-content;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
    position: relative;
}

.category-tab-compact:hover {
    background: #E0E7FF;
    border-color: #A5B4FC;
    transform: translateY(-1px);
}

.category-tab-compact:active {
    transform: scale(0.96);
}

.category-tab-compact.active {
    background: linear-gradient(135deg, #6366F1, #4F46E5);
    color: white;
    border-color: #6366F1;
    box-shadow: 0 2px 12px rgba(99, 102, 241, 0.4);
    transform: translateY(0);
}

.category-tab-compact.active:hover {
    background: linear-gradient(135deg, #4F46E5, #7C3AED);
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.5);
}

/* Compact Cards */
.waiter-card-compact {
    background: linear-gradient(135deg, #FFFFFF 0%, #F8FAFF 100%);
    border-radius: 16px;
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.1), 0 0 0 1px rgba(224, 231, 255, 0.5) inset;
    overflow: hidden;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid rgba(224, 231, 255, 0.8);
    position: relative;
}

.waiter-card-compact::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.waiter-card-compact:hover {
    box-shadow: 0 8px 32px rgba(99, 102, 241, 0.2), 0 0 0 1px rgba(99, 102, 241, 0.3) inset;
    transform: translateY(-2px);
}

.waiter-card-compact:hover::after {
    opacity: 1;
}

.waiter-card-header-compact {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 50%, #7C3AED 100%);
    color: white;
    padding: 0.875rem 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    overflow: hidden;
}

.waiter-card-header-compact::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: shimmer 3s infinite;
}

@keyframes shimmer {
    0%, 100% {
        transform: translate(0, 0) rotate(0deg);
    }
    50% {
        transform: translate(50%, 50%) rotate(180deg);
    }
}

.waiter-card-header-compact h2 {
    margin: 0;
    font-size: 1rem;
    font-weight: 600;
}

.waiter-card-body-compact {
    padding: 1rem;
}

/* Sticky Cart */
.waiter-cart-sticky {
    position: sticky;
    top: 0;
    z-index: 80;
}

/* Products Grid - Tablet Optimized */
.waiter-products-grid-compact {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(160px, 1fr));
    gap: 1rem;
    max-height: calc(100vh - 300px);
    overflow-y: auto;
    padding: 0.75rem;
    -webkit-overflow-scrolling: touch;
}

.waiter-products-grid-compact .product-item {
    animation: fadeInProduct 0.4s ease backwards;
}

@keyframes fadeInProduct {
    from {
        opacity: 0;
        transform: translateY(10px) scale(0.95);
    }
    to {
        opacity: 1;
        transform: translateY(0) scale(1);
    }
}

.waiter-products-grid-compact .product-item:nth-child(1) { animation-delay: 0.05s; }
.waiter-products-grid-compact .product-item:nth-child(2) { animation-delay: 0.1s; }
.waiter-products-grid-compact .product-item:nth-child(3) { animation-delay: 0.15s; }
.waiter-products-grid-compact .product-item:nth-child(4) { animation-delay: 0.2s; }
.waiter-products-grid-compact .product-item:nth-child(5) { animation-delay: 0.25s; }
.waiter-products-grid-compact .product-item:nth-child(6) { animation-delay: 0.3s; }

.waiter-product-item-compact {
    background: linear-gradient(135deg, #F0F4FF 0%, #FFFFFF 50%, #F8FAFF 100%);
    padding: 1rem;
    border-radius: 12px;
    text-align: left;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 2px solid #E0E7FF;
    cursor: pointer;
    display: flex;
    flex-direction: column;
    justify-content: space-between;
    min-height: 160px;
    position: relative;
    overflow: hidden;
}

.waiter-product-item-compact::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 3px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    transform: scaleX(0);
    transform-origin: left;
    transition: transform 0.3s ease;
}

.waiter-product-item-compact:hover {
    transform: translateY(-4px);
    border-color: #6366F1;
    box-shadow: 0 8px 24px rgba(99, 102, 241, 0.25);
    background: linear-gradient(135deg, #FFFFFF 0%, #F0F4FF 50%, #FFFFFF 100%);
}

.waiter-product-item-compact:hover::before {
    transform: scaleX(1);
}

.waiter-product-item-compact:active {
    transform: translateY(-2px) scale(0.98);
}

.waiter-product-item-compact .product-info {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.waiter-product-item-compact .product-header-info {
    display: flex;
    flex-direction: column;
    gap: 0.375rem;
}

.waiter-product-item-compact .product-info h3 {
    font-size: 0.9375rem;
    margin: 0;
    color: #1F2937;
    font-weight: 700;
    line-height: 1.3;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.product-category-badge {
    display: inline-block;
    padding: 0.25rem 0.625rem;
    background: linear-gradient(135deg, #E0E7FF, #C7D2FE);
    color: #4F46E5;
    border-radius: 6px;
    font-size: 0.6875rem;
    font-weight: 600;
    width: fit-content;
    border: 1px solid #C7D2FE;
}

.waiter-product-item-compact .product-code {
    font-size: 0.75rem;
    color: #6B7280;
    margin: 0;
    font-family: 'Courier New', monospace;
}

.waiter-product-item-compact .product-price-section {
    margin: 0.5rem 0;
}

.waiter-product-item-compact .product-price {
    font-size: 1.125rem;
    font-weight: 800;
    background: linear-gradient(135deg, #6366F1, #8B5CF6);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
    line-height: 1.2;
}

.product-stock-info {
    margin-top: 0.5rem;
    margin-bottom: 0.75rem;
}

.stock-indicator {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.stock-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.375rem;
    padding: 0.375rem 0.625rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    width: fit-content;
    line-height: 1;
}

.stock-icon {
    font-size: 0.875rem;
    line-height: 1;
    display: flex;
    align-items: center;
    justify-content: center;
}

.stock-label {
    font-size: 0.75rem;
    font-weight: 600;
    white-space: nowrap;
}

/* Stock Empty */
.stock-indicator.stock-empty .stock-badge {
    background: linear-gradient(135deg, #FEE2E2, #FECACA);
    color: #DC2626;
    border: 1px solid #FCA5A5;
}

.stock-indicator.stock-empty .stock-icon {
    color: #DC2626;
}

/* Stock Low */
.stock-indicator.stock-low .stock-badge {
    background: linear-gradient(135deg, #FEF3C7, #FDE68A);
    color: #D97706;
    border: 1px solid #FCD34D;
}

.stock-indicator.stock-low .stock-icon {
    color: #D97706;
}

/* Stock OK - Clean and Minimal */
.stock-indicator.stock-ok .stock-badge {
    background: linear-gradient(135deg, #D1FAE5, #A7F3D0);
    color: #059669;
    border: 1px solid #6EE7B7;
    padding: 0.25rem 0.5rem;
}

.stock-indicator.stock-ok .stock-badge-success {
    background: linear-gradient(135deg, #D1FAE5 0%, #A7F3D0 100%);
    box-shadow: 0 1px 3px rgba(16, 185, 129, 0.2);
}

.stock-indicator.stock-ok .stock-icon {
    color: #059669;
    font-size: 0.75rem;
    font-weight: 700;
}

.stock-indicator.stock-ok .stock-label {
    color: #059669;
    font-weight: 600;
}

.stock-bar {
    width: 100%;
    height: 4px;
    background: #E5E7EB;
    border-radius: 2px;
    overflow: hidden;
    position: relative;
}

.stock-fill {
    height: 100%;
    border-radius: 2px;
    transition: width 0.5s ease;
    position: relative;
}

.stock-indicator.stock-empty .stock-fill {
    background: linear-gradient(90deg, #EF4444, #DC2626);
    width: 0% !important;
}

.stock-indicator.stock-low .stock-fill {
    background: linear-gradient(90deg, #F59E0B, #D97706);
}

.stock-indicator.stock-ok .stock-fill {
    background: linear-gradient(90deg, #10B981, #059669);
}

.btn-waiter-add {
    background: linear-gradient(135deg, #6366F1 0%, #8B5CF6 50%, #EC4899 100%);
    color: white;
    border: none;
    padding: 0.75rem;
    border-radius: 10px;
    font-weight: 700;
    font-size: 1rem;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    width: 100%;
    min-height: 48px;
    margin-top: 0.75rem;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.35);
    position: relative;
    overflow: hidden;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.btn-waiter-add::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 0;
    height: 0;
    border-radius: 50%;
    background: rgba(255, 255, 255, 0.3);
    transform: translate(-50%, -50%);
    transition: width 0.6s, height 0.6s;
}

.btn-waiter-add:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(99, 102, 241, 0.5);
    background: linear-gradient(135deg, #8B5CF6 0%, #EC4899 50%, #F43F5E 100%);
}

.btn-waiter-add:active:not(:disabled) {
    transform: translateY(0) scale(0.98);
    box-shadow: 0 2px 8px rgba(99, 102, 241, 0.4);
}

.btn-waiter-add:active:not(:disabled)::before {
    width: 300px;
    height: 300px;
}

.btn-waiter-add .btn-icon {
    font-size: 1.25rem;
    font-weight: 800;
    line-height: 1;
}

.btn-waiter-add .btn-text {
    font-size: 0.875rem;
}

.btn-waiter-add:disabled {
    background: #ccc;
    cursor: not-allowed;
    opacity: 0.6;
}

/* Cart Items */
.waiter-cart-items {
    max-height: 220px;
    overflow-y: auto;
    -webkit-overflow-scrolling: touch;
    padding: 0.25rem;
}

.cart-items-list {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.cart-item-card {
    background: linear-gradient(135deg, #F8FAFF 0%, #FFFFFF 50%, #F0F4FF 100%);
    border: 2px solid #E0E7FF;
    border-radius: 10px;
    padding: 0.625rem 0.75rem;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    animation: slideInCart 0.3s ease;
}


.cart-item-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: linear-gradient(180deg, #6366F1, #8B5CF6, #EC4899);
    transform: scaleY(0);
    transform-origin: top;
    transition: transform 0.3s ease;
}

.cart-item-card:hover {
    border-color: #6366F1;
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.2);
    transform: translateX(4px);
}

.cart-item-card:hover::before {
    transform: scaleY(1);
}

.cart-item-info {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.cart-item-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 0.5rem;
}

.cart-item-name-section {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.cart-item-name {
    font-size: 0.875rem;
    color: #1F2937;
    font-weight: 700;
    line-height: 1.3;
}

.btn-remove {
    flex-shrink: 0;
    width: 28px;
    height: 28px;
    padding: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.125rem;
    line-height: 1;
}

.cart-item-code {
    font-size: 0.6875rem;
    color: #6B7280;
    font-family: 'Courier New', monospace;
}

.cart-item-controls {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 0.75rem;
    padding-top: 0.5rem;
    border-top: 1px solid #E0E7FF;
}

.quantity-control {
    display: flex;
    align-items: center;
    gap: 0.375rem;
    background: #F8FAFF;
    padding: 0.25rem;
    border-radius: 6px;
    border: 1px solid #E0E7FF;
}

.quantity-control button {
    background: linear-gradient(135deg, #6366F1, #4F46E5);
    color: white;
    border: none;
    width: 32px;
    height: 32px;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 700;
    font-size: 1rem;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    min-width: 32px;
    min-height: 32px;
    box-shadow: 0 2px 6px rgba(99, 102, 241, 0.3);
    display: flex;
    align-items: center;
    justify-content: center;
    line-height: 1;
}

.quantity-control button:hover {
    transform: scale(1.1);
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.4);
    background: linear-gradient(135deg, #8B5CF6, #6366F1);
}

.quantity-control button:active {
    transform: scale(0.95);
}

.quantity {
    min-width: 28px;
    text-align: center;
    font-weight: 700;
    color: #1F2937;
    font-size: 0.875rem;
    padding: 0 0.375rem;
}

.cart-item-subtotal {
    display: flex;
    align-items: center;
}

.subtotal-amount {
    font-size: 0.9375rem;
    color: #6366F1;
    font-weight: 800;
    background: linear-gradient(135deg, #6366F1, #8B5CF6);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.btn-waiter-clear {
    background: rgba(255,255,255,0.2);
    color: white;
    border: 2px solid rgba(255,255,255,0.5);
    padding: 0.375rem 0.625rem;
    border-radius: 6px;
    font-size: 1rem;
    cursor: pointer;
    transition: all 0.2s;
    min-width: 44px;
    min-height: 36px;
}

.btn-waiter-clear:active {
    background: white;
    color: #6366F1;
    transform: scale(0.95);
}

.waiter-cart-summary-compact {
    margin-top: 0.75rem;
    padding-top: 0.75rem;
    border-top: 2px solid #E0E7FF;
    background: linear-gradient(135deg, #F8FAFF, #FFFFFF);
    padding: 1rem;
    border-radius: 10px;
    border: 2px solid #E0E7FF;
}

.waiter-cart-summary-compact .summary-row {
    display: flex;
    justify-content: space-between;
    padding: 0.625rem 0;
    font-size: 0.9375rem;
    color: #4B5563;
    transition: all 0.2s ease;
}

.waiter-cart-summary-compact .summary-row:hover {
    color: #1F2937;
}

.waiter-cart-summary-compact .summary-row.total {
    border-top: 2px solid #6366F1;
    margin-top: 0.75rem;
    padding-top: 0.875rem;
    font-size: 1.25rem;
    background: linear-gradient(135deg, #6366F1, #8B5CF6);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    font-weight: 800;
    position: relative;
}

.waiter-cart-summary-compact .summary-row.total::after {
    content: '';
    position: absolute;
    bottom: -0.5rem;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    border-radius: 1px;
}

/* Payment Form - Compact & Clean */
.payment-form-compact {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.form-group-compact {
    margin-bottom: 0;
}

.form-label-compact {
    display: block;
    margin-bottom: 0.375rem;
    font-weight: 600;
    font-size: 0.8125rem;
    color: #4B5563;
}

.form-info-compact {
    display: block;
    margin-top: 0.375rem;
    font-size: 0.6875rem;
    color: #6B7280;
}

.table-selection-compact {
    max-height: 140px;
    overflow-y: auto;
    border: 1.5px solid #E0E7FF;
    border-radius: 8px;
    background: #F8FAFF;
    padding: 0.5rem;
    -webkit-overflow-scrolling: touch;
}

.table-option-compact {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s;
    margin-bottom: 0.25rem;
    min-height: 36px;
    font-size: 0.8125rem;
}

.table-option-compact:hover {
    background: #E0E7FF;
}

.table-option-compact:active {
    background: #C7D2FE;
}

.table-option-compact input[type="checkbox"] {
    width: 18px;
    height: 18px;
    cursor: pointer;
    flex-shrink: 0;
}

.table-option-compact span {
    flex: 1;
    font-weight: 500;
    color: #374151;
}

.table-option-compact span small {
    color: #9CA3AF;
    font-size: 0.75rem;
    margin-left: 0.25rem;
}

.checkbox-compact {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 6px;
    transition: all 0.2s;
    min-height: 36px;
    background: #F8FAFF;
    border: 1.5px solid #E0E7FF;
}

.checkbox-compact:hover {
    background: #E0E7FF;
    border-color: #C7D2FE;
}

.checkbox-compact input[type="checkbox"] {
    width: 18px;
    height: 18px;
    cursor: pointer;
    flex-shrink: 0;
}

.checkbox-compact span {
    font-size: 0.8125rem;
    font-weight: 600;
    color: #374151;
}

.payment-method-section {
    padding: 0.75rem;
    background: #F8FAFF;
    border-radius: 8px;
    border: 1.5px solid #E0E7FF;
}

.form-row-compact {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 0.75rem;
}

.form-input-compact {
    border: 1.5px solid #E0E7FF;
    border-radius: 6px;
    padding: 0.625rem 0.75rem;
    font-size: 0.875rem;
    transition: all 0.2s;
    width: 100%;
    min-height: 38px;
    background: white;
    font-weight: 500;
}

.form-input-compact:focus {
    outline: none;
    border-color: #6366F1;
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.btn-checkout-compact {
    background: linear-gradient(135deg, #10B981, #059669);
    color: white;
    border: none;
    font-weight: 700;
    font-size: 1rem;
    padding: 0.875rem;
    border-radius: 8px;
    transition: all 0.2s;
    min-height: 48px;
    margin-top: 0.5rem;
    width: 100%;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
    cursor: pointer;
}

.btn-checkout-compact:hover:not(:disabled) {
    background: linear-gradient(135deg, #059669, #047857);
    box-shadow: 0 6px 16px rgba(16, 185, 129, 0.4);
    transform: translateY(-1px);
}

.btn-checkout-compact:active:not(:disabled) {
    transform: translateY(0) scale(0.98);
}

.btn-checkout-compact:disabled {
    background: #D1D5DB;
    cursor: not-allowed;
    opacity: 0.6;
    box-shadow: none;
}

/* Form Controls - Compact (Legacy support) */
.waiter-form-group-compact {
    margin-bottom: 1rem;
}

.waiter-label-compact {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    font-size: 0.875rem;
    color: #2D3436;
}

.waiter-table-selection {
    max-height: 180px;
    overflow-y: auto;
    border: 2px solid #E0E7FF;
    border-radius: 8px;
    background: #F8FAFF;
    padding: 0.5rem;
    -webkit-overflow-scrolling: touch;
}

.waiter-table-option {
    display: block;
    padding: 0.625rem;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s;
    margin-bottom: 0.375rem;
    min-height: 44px;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.waiter-table-option:active {
    background: #E0E7FF;
}

.waiter-table-option input[type="checkbox"] {
    width: 20px;
    height: 20px;
    cursor: pointer;
}

.waiter-table-option span {
    flex: 1;
    font-size: 0.875rem;
}

.waiter-table-option span small {
    color: #999;
    font-size: 0.75rem;
}

.waiter-table-info {
    display: block;
    margin-top: 0.5rem;
    font-size: 0.75rem;
    color: #666;
}

.waiter-checkbox-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
    padding: 0.625rem;
    border-radius: 6px;
    transition: all 0.2s;
    min-height: 44px;
}

.waiter-checkbox-label:active {
    background: #F8FAFF;
}

.waiter-checkbox-label input[type="checkbox"] {
    width: 20px;
    height: 20px;
    cursor: pointer;
}

.waiter-checkbox-label span {
    font-size: 0.875rem;
}

.waiter-select-compact,
.waiter-input-compact {
    border: 2px solid #E0E7FF;
    border-radius: 8px;
    padding: 0.75rem;
    font-size: 1rem;
    transition: all 0.2s;
    width: 100%;
    min-height: 44px;
    background: #F8FAFF;
}

.waiter-select-compact:focus,
.waiter-input-compact:focus {
    outline: none;
    border-color: #6366F1;
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
    background: white;
}

/* Checkout Button */
.btn-waiter-checkout {
    background: linear-gradient(135deg, #10B981, #059669);
    color: white;
    border: none;
    font-weight: 700;
    font-size: 1.125rem;
    padding: 1rem;
    border-radius: 10px;
    transition: all 0.2s;
    min-height: 56px;
    margin-top: 0.75rem;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
}

.btn-waiter-checkout:active:not(:disabled) {
    transform: scale(0.98);
    background: linear-gradient(135deg, #059669, #047857);
    box-shadow: 0 2px 8px rgba(16, 185, 129, 0.4);
}

.btn-waiter-checkout:disabled {
    background: #ccc;
    cursor: not-allowed;
    opacity: 0.6;
}

/* Scrollbar Styling */
.waiter-products-grid-compact::-webkit-scrollbar,
.waiter-table-selection::-webkit-scrollbar,
.waiter-cart-items::-webkit-scrollbar,
.table-selection-compact::-webkit-scrollbar {
    width: 6px;
}

.waiter-products-grid-compact::-webkit-scrollbar-track,
.waiter-table-selection::-webkit-scrollbar-track,
.waiter-cart-items::-webkit-scrollbar-track,
.table-selection-compact::-webkit-scrollbar-track {
    background: #F8FAFF;
    border-radius: 3px;
}

.waiter-products-grid-compact::-webkit-scrollbar-thumb,
.waiter-table-selection::-webkit-scrollbar-thumb,
.waiter-cart-items::-webkit-scrollbar-thumb,
.table-selection-compact::-webkit-scrollbar-thumb {
    background: #6366F1;
    border-radius: 3px;
}

.waiter-products-grid-compact::-webkit-scrollbar-thumb:hover,
.waiter-table-selection::-webkit-scrollbar-thumb:hover,
.waiter-cart-items::-webkit-scrollbar-thumb:hover,
.table-selection-compact::-webkit-scrollbar-thumb:hover {
    background: #4F46E5;
}

/* Fullscreen Mode */
body.waiter-fullscreen-active .waiter-pos-container-compact {
    max-width: 100%;
    padding: 0.5rem;
}

body.waiter-fullscreen-active .navbar {
    display: none;
}

body.waiter-fullscreen-active .container {
    max-width: 100%;
    margin: 0;
    padding: 0.5rem;
}

body.waiter-fullscreen-active .waiter-pos-header-compact {
    margin-bottom: 0.75rem;
    padding: 0.625rem 0.875rem;
}

/* Tablet Optimizations */
@media (min-width: 768px) and (max-width: 1024px) {
    .waiter-pos-container-compact {
        grid-template-columns: 1.75fr 1fr;
    }
    
    .waiter-products-grid-compact {
        grid-template-columns: repeat(auto-fill, minmax(130px, 1fr));
    }
}

/* Mobile Responsive */
@media (max-width: 767px) {
    .waiter-pos-container-compact {
        grid-template-columns: 1fr;
    }
    
    .waiter-pos-header-compact {
        flex-direction: column;
        gap: 0.5rem;
        text-align: center;
    }
    
    .waiter-products-grid-compact {
        grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
    }
    
    .waiter-cart-sticky {
        position: relative;
    }
    
    .form-row-compact {
        grid-template-columns: 1fr;
    }
    
    .payment-method-section {
        padding: 0.5rem;
    }
}

/* Cart Buttons */
.btn-waiter-cart {
    background: #6366F1;
    color: white;
    border: none;
    min-width: 36px;
    min-height: 36px;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 700;
    font-size: 1rem;
    transition: all 0.2s;
    box-shadow: 0 2px 4px rgba(99, 102, 241, 0.2);
}

.btn-waiter-cart:active {
    background: #4F46E5;
    transform: scale(0.9);
}

.btn-waiter-danger {
    background: #FF4757;
    color: white;
}

.btn-waiter-danger:active {
    background: #FF3838;
    transform: scale(0.9);
}

/* Touch-friendly improvements */
@media (hover: none) and (pointer: coarse) {
    .waiter-product-item-compact:hover {
        transform: none;
    }
    
    .category-tab-compact:hover {
        transform: none;
    }
    
    .btn-waiter-add:hover {
        transform: none;
    }
}
</style>
<?php endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>

