/**
 * Service Worker untuk POS Offline-First
 * Menangani caching dan background sync
 */

const CACHE_NAME = 'pos-offline-v1';
const STATIC_CACHE = 'pos-static-v1';
const DYNAMIC_CACHE = 'pos-dynamic-v1';

// Assets yang harus di-cache saat install
const STATIC_ASSETS = [
  '/pos/public/pos.php',
  '/pos/public/index.php',
  '/pos/public/assets/css/style.css',
  '/pos/public/assets/js/main.js',
  '/pos/public/assets/js/offline-db.js',
  '/pos/public/assets/js/sync-engine.js',
  '/pos/public/manifest.json'
];

// Install event - Cache static assets
self.addEventListener('install', (event) => {
  console.log('[SW] Installing Service Worker...');
  
  event.waitUntil(
    caches.open(STATIC_CACHE).then((cache) => {
      console.log('[SW] Caching static assets');
      return cache.addAll(STATIC_ASSETS.map(url => new Request(url, {credentials: 'same-origin'})));
    }).catch(err => {
      console.error('[SW] Cache install failed:', err);
    })
  );
  
  // Force activation
  self.skipWaiting();
});

// Activate event - Clean old caches
self.addEventListener('activate', (event) => {
  console.log('[SW] Activating Service Worker...');
  
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== STATIC_CACHE && cacheName !== DYNAMIC_CACHE) {
            console.log('[SW] Deleting old cache:', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    })
  );
  
  // Take control of all pages
  return self.clients.claim();
});

// Fetch event - Network first dengan fallback ke cache
self.addEventListener('fetch', (event) => {
  const { request } = event;
  const url = new URL(request.url);
  
  // Skip non-GET requests
  if (request.method !== 'GET') {
    return;
  }
  
  // Skip API calls (akan ditangani oleh sync engine)
  if (url.pathname.includes('/api/')) {
    return;
  }
  
  // Strategy: Network first, fallback to cache
  event.respondWith(
    fetch(request)
      .then((response) => {
        // Clone response untuk cache
        const responseToCache = response.clone();
        
        // Cache dynamic content
        if (url.origin === location.origin) {
          caches.open(DYNAMIC_CACHE).then((cache) => {
            cache.put(request, responseToCache);
          });
        }
        
        return response;
      })
      .catch(() => {
        // Network failed, try cache
        return caches.match(request).then((cachedResponse) => {
          if (cachedResponse) {
            return cachedResponse;
          }
          
          // If HTML page, return offline page
          if (request.headers.get('accept').includes('text/html')) {
            return caches.match('/pos/public/pos.php');
          }
          
          // Return offline response
          return new Response('Offline', {
            status: 503,
            statusText: 'Service Unavailable',
            headers: new Headers({
              'Content-Type': 'text/plain'
            })
          });
        });
      })
  );
});

// Background Sync untuk sinkronisasi transaksi
self.addEventListener('sync', (event) => {
  console.log('[SW] Background sync triggered:', event.tag);
  
  if (event.tag === 'sync-transactions') {
    event.waitUntil(syncPendingTransactions());
  }
});

// Sync pending transactions
async function syncPendingTransactions() {
  try {
    // Post message ke client untuk trigger sync
    const clients = await self.clients.matchAll();
    clients.forEach(client => {
      client.postMessage({
        type: 'SYNC_TRANSACTIONS',
        timestamp: Date.now()
      });
    });
  } catch (error) {
    console.error('[SW] Sync error:', error);
  }
}

// Message handler dari client
self.addEventListener('message', (event) => {
  if (event.data && event.data.type === 'SKIP_WAITING') {
    self.skipWaiting();
  }
  
  if (event.data && event.data.type === 'CACHE_URLS') {
    event.waitUntil(
      caches.open(DYNAMIC_CACHE).then((cache) => {
        return cache.addAll(event.data.urls);
      })
    );
  }
});

// Push notification handler (optional untuk future)
self.addEventListener('push', (event) => {
  console.log('[SW] Push notification received');
  // Implementasi push notification jika diperlukan
});

