<?php
/**
 * Tables Management
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();
$auth = new Auth();
$auth->requirePermission('manager');

$pageTitle = 'Manajemen Meja';

$table = new Table();

$transaction = new Transaction();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action == 'create') {
        $name = $_POST['name'] ?? '';
        $capacity = intval($_POST['capacity'] ?? 4);
        $location = $_POST['location'] ?? '';
        $notes = $_POST['notes'] ?? '';
        if ($table->create($name, $capacity, $location, $notes)) {
            setFlash('success', 'Meja berhasil ditambahkan!');
        } else {
            setFlash('error', 'Gagal menambahkan meja!');
        }
        redirect(BASE_URL . 'public/tables.php');
    }
    
    if ($action == 'merge_tables') {
        $source_table_id = intval($_POST['source_table_id'] ?? 0);
        $target_table_ids = isset($_POST['target_table_ids']) ? array_map('intval', $_POST['target_table_ids']) : [];
        
        if ($source_table_id > 0 && !empty($target_table_ids)) {
            // Get source table's transaction (if exists)
            $source_transaction = $table->getTableTransaction($source_table_id);
            
            if ($source_transaction) {
                // Source meja sudah punya transaksi - gabungkan meja lain ke transaksi ini
                if ($transaction->mergeTables($source_transaction['id'], $target_table_ids)) {
                    setFlash('success', 'Meja berhasil digabung dengan transaksi yang sudah ada!');
                } else {
                    setFlash('error', 'Gagal menggabung meja!');
                }
            } else {
                // Source meja belum punya transaksi - buat group meja (reserved/merged state)
                // Semua meja akan ditandai sebagai "reserved" dan siap untuk transaksi bersama
                $all_table_ids = array_merge([$source_table_id], $target_table_ids);
                $success = true;
                
                foreach ($all_table_ids as $table_id) {
                    // Update status menjadi reserved untuk menandai meja sudah di-merge
                    if (!$table->updateStatus($table_id, 'reserved')) {
                        $success = false;
                        break;
                    }
                }
                
                if ($success) {
                    // Simpan informasi merge (bisa digunakan saat checkout nanti)
                    // Untuk sekarang, kita hanya tandai status sebagai reserved
                    // Saat checkout di POS, user bisa pilih beberapa meja yang sudah di-reserve ini
                    setFlash('success', 'Meja berhasil digabung! Meja akan ditandai sebagai "Dipesan" dan siap digunakan bersama saat checkout.');
                } else {
                    setFlash('error', 'Gagal menggabung meja!');
                }
            }
        } else {
            setFlash('error', 'Pilih meja yang akan digabung!');
        }
        redirect(BASE_URL . 'public/tables.php');
    }
    
    if ($action == 'unmerge_table') {
        $transaction_id = intval($_POST['transaction_id'] ?? 0);
        $table_id = intval($_POST['table_id'] ?? 0);
        
        if ($transaction_id > 0 && $table_id > 0) {
            if ($transaction->unmergeTable($transaction_id, $table_id)) {
                setFlash('success', 'Meja berhasil dipisahkan!');
            } else {
                setFlash('error', 'Gagal memisahkan meja! Pastikan masih ada meja lain yang terhubung.');
            }
        }
        redirect(BASE_URL . 'public/tables.php');
    }
    
    if ($action == 'unmerge_group') {
        // Unmerge meja yang di-group tanpa transaksi (status reserved)
        $table_id = intval($_POST['table_id'] ?? 0);
        
        if ($table_id > 0) {
            // Check if table is reserved and has no active transaction
            $table_info = $table->getById($table_id);
            if ($table_info && $table_info['status'] == 'reserved') {
                $table_status = $table->getTableStatus($table_id);
                if (!$table_status || !$table_status['current_transaction_id']) {
                    // No active transaction, can unmerge by setting status back to available
                    if ($table->updateStatus($table_id, 'available')) {
                        setFlash('success', 'Meja berhasil dipisahkan dari group!');
                    } else {
                        setFlash('error', 'Gagal memisahkan meja!');
                    }
                } else {
                    setFlash('error', 'Meja ini memiliki transaksi aktif. Gunakan tombol "Pisahkan" di bagian transaksi.');
                }
            } else {
                setFlash('error', 'Meja ini tidak dalam status "Dipesan" atau sudah memiliki transaksi.');
            }
        }
        redirect(BASE_URL . 'public/tables.php');
    }
    
    if ($action == 'update') {
        $id = $_POST['id'] ?? 0;
        $name = $_POST['name'] ?? '';
        $capacity = intval($_POST['capacity'] ?? 4);
        $location = $_POST['location'] ?? '';
        $notes = $_POST['notes'] ?? '';
        if ($table->update($id, $name, $capacity, $location, $notes)) {
            setFlash('success', 'Meja berhasil diupdate!');
        } else {
            setFlash('error', 'Gagal mengupdate meja!');
        }
        redirect(BASE_URL . 'public/tables.php');
    }
    
    if ($action == 'update_status') {
        $id = $_POST['id'] ?? 0;
        $status = $_POST['status'] ?? 'available';
        if ($table->updateStatus($id, $status)) {
            setFlash('success', 'Status meja berhasil diupdate!');
        } else {
            setFlash('error', 'Gagal mengupdate status meja!');
        }
        redirect(BASE_URL . 'public/tables.php');
    }
    
    if ($action == 'delete') {
        $id = $_POST['id'] ?? 0;
        if ($table->delete($id)) {
            setFlash('success', 'Meja berhasil dihapus!');
        } else {
            setFlash('error', 'Gagal menghapus meja! Meja mungkin masih memiliki transaksi aktif.');
        }
        redirect(BASE_URL . 'public/tables.php');
    }
}

$tables = $table->getAll();
$transaction = new Transaction();

// Get table status with merged info for ALL tables (especially occupied ones)
foreach ($tables as &$tbl) {
    // Always check table status, especially for occupied tables
    if ($tbl['status'] == 'occupied') {
        $table_status = $table->getTableStatus($tbl['id']);
        if ($table_status && $table_status['current_transaction_id']) {
            $merged_tables = $transaction->getTransactionTables($table_status['current_transaction_id']);
            $tbl['merged_tables'] = $merged_tables;
            $tbl['current_transaction_id'] = $table_status['current_transaction_id'];
            $tbl['is_merged'] = count($merged_tables) > 1;
            $tbl['invoice_number'] = $table_status['invoice_number'] ?? null;
        } else {
            // Try to get transaction via old method (table_id directly)
            $old_transaction = $table->getTableTransaction($tbl['id']);
            if ($old_transaction) {
                $tbl['current_transaction_id'] = $old_transaction['id'];
                $tbl['invoice_number'] = $old_transaction['invoice_number'];
                // Check if already merged
                $merged_tables = $transaction->getTransactionTables($old_transaction['id']);
                $tbl['merged_tables'] = $merged_tables;
                $tbl['is_merged'] = count($merged_tables) > 1;
            }
        }
    } else {
        // For non-occupied tables, still check if they have transaction (might be in transition)
        $table_status = $table->getTableStatus($tbl['id']);
        if ($table_status && $table_status['current_transaction_id']) {
            $merged_tables = $transaction->getTransactionTables($table_status['current_transaction_id']);
            $tbl['merged_tables'] = $merged_tables;
            $tbl['current_transaction_id'] = $table_status['current_transaction_id'];
            $tbl['is_merged'] = count($merged_tables) > 1;
        }
    }
}

include __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1>🪑 Manajemen Meja</h1>
    <div style="display: flex; gap: 1rem;">
        <button class="btn btn-primary" onclick="showModal('createModal')">+ Tambah Meja</button>
        <button class="btn btn-warning" onclick="showMergeModal()">🔗 Merge Meja</button>
    </div>
</div>

<div class="tables-grid">
    <?php foreach ($tables as $tbl): ?>
        <?php
        $status_class = '';
        $status_label = '';
        switch ($tbl['status']) {
            case 'available':
                $status_class = 'status-available';
                $status_label = 'Kosong';
                break;
            case 'occupied':
                $status_class = 'status-occupied';
                $status_label = 'Terisi';
                break;
            case 'reserved':
                $status_class = 'status-reserved';
                $status_label = 'Dipesan';
                break;
            case 'cleaning':
                $status_class = 'status-cleaning';
                $status_label = 'Bersihkan';
                break;
        }
        ?>
        <div class="table-card <?php echo $status_class; ?>">
            <div class="table-card-header">
                <h3><?php echo escape($tbl['name']); ?></h3>
                <span class="table-status-badge status-<?php echo $tbl['status']; ?>">
                    <?php echo $status_label; ?>
                </span>
            </div>
            <div class="table-card-body">
                <div class="table-info">
                    <p><strong>Kapasitas:</strong> <?php echo $tbl['capacity']; ?> orang</p>
                    <?php if ($tbl['location']): ?>
                        <p><strong>Lokasi:</strong> <?php echo escape($tbl['location']); ?></p>
                    <?php endif; ?>
                    <?php if ($tbl['notes']): ?>
                        <p><strong>Catatan:</strong> <?php echo escape($tbl['notes']); ?></p>
                    <?php endif; ?>
                    <?php if ($tbl['active_transactions'] > 0): ?>
                        <p><small class="text-muted">Transaksi hari ini: <?php echo $tbl['active_transactions']; ?></small></p>
                    <?php endif; ?>
                    <?php if (!empty($tbl['is_merged']) && !empty($tbl['merged_tables'])): ?>
                        <div style="margin-top: 0.5rem; padding: 0.5rem; background: #fff3cd; border-radius: 4px;">
                            <strong style="color: #856404;">🔗 Digabung dengan:</strong>
                            <?php foreach ($tbl['merged_tables'] as $merged): ?>
                                <?php if ($merged['id'] != $tbl['id']): ?>
                                    <span class="badge badge-warning" style="margin-left: 5px;">
                                        <?php echo escape($merged['name']); ?>
                                    </span>
                                <?php endif; ?>
                            <?php endforeach; ?>
                            <?php if (!empty($tbl['current_transaction_id'])): ?>
                                <form method="POST" style="display: inline; margin-left: 10px;" 
                                      onsubmit="return confirm('Yakin ingin memisahkan meja ini?');">
                                    <input type="hidden" name="action" value="unmerge_table">
                                    <input type="hidden" name="transaction_id" value="<?php echo $tbl['current_transaction_id']; ?>">
                                    <input type="hidden" name="table_id" value="<?php echo $tbl['id']; ?>">
                                    <button type="submit" class="btn btn-xs btn-danger">Pisahkan</button>
                                </form>
                            <?php endif; ?>
                        </div>
                    <?php elseif ($tbl['status'] == 'reserved' && empty($tbl['current_transaction_id'])): ?>
                        <div style="margin-top: 0.5rem; padding: 0.5rem; background: #d1ecf1; border-radius: 4px;">
                            <strong style="color: #0c5460;">📋 Status: Dipesan (Group)</strong>
                            <small class="text-muted">Meja ini sudah di-group dan siap digunakan bersama saat checkout</small>
                            <form method="POST" style="display: inline; margin-left: 10px;" 
                                  onsubmit="return confirm('Yakin ingin membatalkan group meja ini?');">
                                <input type="hidden" name="action" value="unmerge_group">
                                <input type="hidden" name="table_id" value="<?php echo $tbl['id']; ?>">
                                <button type="submit" class="btn btn-xs btn-secondary">Batal Group</button>
                            </form>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="table-actions">
                    <button class="btn btn-xs btn-primary" onclick="editTable(<?php echo htmlspecialchars(json_encode($tbl)); ?>)">Edit</button>
                    <div class="status-dropdown">
                        <select class="form-control form-control-sm" onchange="updateTableStatus(<?php echo $tbl['id']; ?>, this.value)">
                            <option value="available" <?php echo $tbl['status'] == 'available' ? 'selected' : ''; ?>>Kosong</option>
                            <option value="occupied" <?php echo $tbl['status'] == 'occupied' ? 'selected' : ''; ?>>Terisi</option>
                            <option value="reserved" <?php echo $tbl['status'] == 'reserved' ? 'selected' : ''; ?>>Dipesan</option>
                            <option value="cleaning" <?php echo $tbl['status'] == 'cleaning' ? 'selected' : ''; ?>>Bersihkan</option>
                        </select>
                    </div>
                    <form method="POST" style="display:inline;" onsubmit="return confirm('Yakin hapus meja ini?');">
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="id" value="<?php echo $tbl['id']; ?>">
                        <button type="submit" class="btn btn-xs btn-danger">Hapus</button>
                    </form>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<!-- Create Modal -->
<div id="createModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>Tambah Meja</h2>
            <span class="close" onclick="closeModal('createModal')">&times;</span>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="create">
            <div class="modal-body">
                <div class="form-group">
                    <label>Nama Meja *</label>
                    <input type="text" name="name" required placeholder="Contoh: Meja 1">
                </div>
                <div class="form-group">
                    <label>Kapasitas *</label>
                    <input type="number" name="capacity" min="1" max="20" value="4" required>
                </div>
                <div class="form-group">
                    <label>Lokasi</label>
                    <input type="text" name="location" placeholder="Contoh: Area Depan">
                </div>
                <div class="form-group">
                    <label>Catatan</label>
                    <textarea name="notes" rows="2" placeholder="Catatan tambahan..."></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('createModal')">Batal</button>
                <button type="submit" class="btn btn-primary">Simpan</button>
            </div>
        </form>
    </div>
</div>

<!-- Merge Tables Modal -->
<div id="mergeModal" class="modal">
    <div class="modal-content" style="max-width: 600px;">
        <div class="modal-header">
            <h2>🔗 Merge / Gabung Meja</h2>
            <span class="close" onclick="closeModal('mergeModal')">&times;</span>
        </div>
        <form method="POST" id="mergeForm">
            <input type="hidden" name="action" value="merge_tables">
            <input type="hidden" name="source_table_id" id="merge_source_table_id">
            <div class="modal-body">
                <div class="form-group">
                    <label>Pilih Meja Sumber *</label>
                    <select name="source_table_id_select" id="merge_source_select" class="form-control" required onchange="updateMergeTargets()">
                        <option value="">-- Pilih Meja Sumber --</option>
                        <?php 
                        // Show all tables except cleaning status
                        $mergeable_source_tables = array_filter($tables, function($t) {
                            return $t['status'] != 'cleaning';
                        });
                        foreach ($mergeable_source_tables as $tbl):
                            $has_transaction = !empty($tbl['current_transaction_id']);
                            $status_label = $has_transaction ? '(Ada Transaksi)' : '(Kosong)';
                        ?>
                            <option value="<?php echo $tbl['id']; ?>" 
                                    data-transaction="<?php echo $tbl['current_transaction_id'] ?? ''; ?>"
                                    data-status="<?php echo $tbl['status']; ?>">
                                <?php echo escape($tbl['name']); ?> 
                                (<?php echo $tbl['capacity']; ?> orang)
                                <?php if ($tbl['location']): ?>
                                    - <?php echo escape($tbl['location']); ?>
                                <?php endif; ?>
                                - <strong><?php echo $status_label; ?></strong>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="text-muted">
                        Pilih meja sumber. Jika meja sudah ada transaksi, meja lain akan digabung ke transaksi tersebut. 
                        Jika meja kosong, semua meja akan di-group untuk digunakan bersama saat checkout.
                    </small>
                </div>
                <div class="form-group">
                    <label>Pilih Meja yang Akan Digabung *</label>
                    <div id="mergeTargets" style="max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; border-radius: 4px; background: #f9f9f9;">
                        <p class="text-muted">Pilih meja sumber terlebih dahulu</p>
                    </div>
                    <small class="text-muted">
                        Pilih satu atau lebih meja yang akan digabung dengan meja sumber. 
                        Bisa memilih meja kosong, terisi, atau dipesan.
                    </small>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('mergeModal')">Batal</button>
                <button type="submit" class="btn btn-warning">Gabung Meja</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>Edit Meja</h2>
            <span class="close" onclick="closeModal('editModal')">&times;</span>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="update">
            <input type="hidden" name="id" id="edit_id">
            <div class="modal-body">
                <div class="form-group">
                    <label>Nama Meja *</label>
                    <input type="text" name="name" id="edit_name" required>
                </div>
                <div class="form-group">
                    <label>Kapasitas *</label>
                    <input type="number" name="capacity" id="edit_capacity" min="1" max="20" required>
                </div>
                <div class="form-group">
                    <label>Lokasi</label>
                    <input type="text" name="location" id="edit_location">
                </div>
                <div class="form-group">
                    <label>Catatan</label>
                    <textarea name="notes" id="edit_notes" rows="2"></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('editModal')">Batal</button>
                <button type="submit" class="btn btn-primary">Update</button>
            </div>
        </form>
    </div>
</div>

<style>
.tables-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-top: 1.5rem;
}

.table-card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    overflow: hidden;
    transition: transform 0.3s, box-shadow 0.3s;
    border: 3px solid #e0e0e0;
}

.table-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 4px 16px rgba(0,0,0,0.15);
}

.table-card.status-available {
    border-color: #28a745;
    background: #f0fff4;
}

.table-card.status-occupied {
    border-color: #dc3545;
    background: #fff5f5;
}

.table-card.status-reserved {
    border-color: #ffc107;
    background: #fffbf0;
}

.table-card.status-cleaning {
    border-color: #17a2b8;
    background: #f0f9ff;
}

.table-card-header {
    padding: 1rem 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 2px solid #e0e0e0;
}

.table-card-header h3 {
    margin: 0;
    font-size: 1.25rem;
    color: #333;
}

.table-status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: bold;
    text-transform: uppercase;
}

.table-status-badge.status-available {
    background: #28a745;
    color: white;
}

.table-status-badge.status-occupied {
    background: #dc3545;
    color: white;
}

.table-status-badge.status-reserved {
    background: #ffc107;
    color: #333;
}

.table-status-badge.status-cleaning {
    background: #17a2b8;
    color: white;
}

.table-card-body {
    padding: 1.5rem;
}

.table-info {
    margin-bottom: 1rem;
}

.table-info p {
    margin: 0.5rem 0;
    font-size: 0.9rem;
}

.table-actions {
    display: flex;
    gap: 0.5rem;
    align-items: center;
    flex-wrap: wrap;
    padding-top: 1rem;
    border-top: 1px solid #e0e0e0;
}

.status-dropdown {
    flex: 1;
    min-width: 120px;
}

.status-dropdown select {
    width: 100%;
}

@media (max-width: 768px) {
    .tables-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
const allTables = <?php echo json_encode($tables); ?>;

function editTable(table) {
    document.getElementById('edit_id').value = table.id;
    document.getElementById('edit_name').value = table.name;
    document.getElementById('edit_capacity').value = table.capacity;
    document.getElementById('edit_location').value = table.location || '';
    document.getElementById('edit_notes').value = table.notes || '';
    showModal('editModal');
}

function updateTableStatus(tableId, status) {
    if (confirm('Yakin ingin mengubah status meja?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="action" value="update_status">
            <input type="hidden" name="id" value="${tableId}">
            <input type="hidden" name="status" value="${status}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}

function showMergeModal() {
    document.getElementById('merge_source_table_id').value = '';
    document.getElementById('merge_source_select').value = '';
    document.getElementById('mergeTargets').innerHTML = '<p class="text-muted">Pilih meja sumber terlebih dahulu</p>';
    showModal('mergeModal');
}

function updateMergeTargets() {
    const sourceSelect = document.getElementById('merge_source_select');
    const sourceTableId = parseInt(sourceSelect.value);
    const mergeTargets = document.getElementById('mergeTargets');
    
    if (!sourceTableId) {
        mergeTargets.innerHTML = '<p class="text-muted">Pilih meja sumber terlebih dahulu</p>';
        document.getElementById('merge_source_table_id').value = '';
        return;
    }
    
    const selectedOption = sourceSelect.options[sourceSelect.selectedIndex];
    const sourceHasTransaction = selectedOption.getAttribute('data-transaction') !== '';
    const sourceStatus = selectedOption.getAttribute('data-status');
    
    document.getElementById('merge_source_table_id').value = sourceTableId;
    
    // Get tables that can be merged (exclude source and already merged)
    const sourceTable = allTables.find(t => t.id == sourceTableId);
    const sourceMergedIds = sourceTable && sourceTable.merged_tables ? 
        sourceTable.merged_tables.map(t => t.id) : [sourceTableId];
    
    // Filter: all tables except source, cleaning, and already merged
    const availableTables = allTables.filter(t => {
        // Not the source table
        if (t.id == sourceTableId) return false;
        // Not cleaning status
        if (t.status === 'cleaning') return false;
        // Not already merged with source
        if (sourceMergedIds.includes(t.id)) return false;
        // If source has transaction, prefer tables without transaction (to merge empty tables to existing transaction)
        // If source doesn't have transaction, show all tables (to create a group)
        return true;
    });
    
    if (availableTables.length === 0) {
        mergeTargets.innerHTML = '<p class="text-warning">Tidak ada meja yang bisa digabung.</p>';
        return;
    }
    
    let html = '';
    if (sourceHasTransaction) {
        html += '<p class="text-info"><small>💡 Meja sumber sudah ada transaksi. Meja yang dipilih akan digabung ke transaksi tersebut.</small></p>';
    } else {
        html += '<p class="text-info"><small>💡 Meja sumber kosong. Meja yang dipilih akan di-group untuk digunakan bersama saat checkout.</small></p>';
    }
    
    // Sort: show available/occupied tables first, then reserved
    availableTables.sort((a, b) => {
        const statusOrder = {'available': 1, 'occupied': 2, 'reserved': 3};
        return (statusOrder[a.status] || 99) - (statusOrder[b.status] || 99);
    });
    
    availableTables.forEach(table => {
        const statusLabels = {
            'available': '<span class="badge badge-success">Kosong</span>',
            'occupied': '<span class="badge badge-danger">Terisi</span>',
            'reserved': '<span class="badge badge-warning">Dipesan</span>'
        };
        const statusLabel = statusLabels[table.status] || '';
        const hasTransaction = table.current_transaction_id ? 
            '<small class="text-success">(Ada transaksi)</small>' : 
            '<small class="text-muted">(Kosong)</small>';
        
        html += `
            <label style="display: block; margin-bottom: 8px; padding: 8px; border-radius: 4px; cursor: pointer; border: 1px solid #ddd; background: white;" 
                   onmouseover="this.style.background='#e9ecef'; this.style.borderColor='#007bff';" 
                   onmouseout="this.style.background='white'; this.style.borderColor='#ddd';">
                <input type="checkbox" name="target_table_ids[]" value="${table.id}" style="margin-right: 8px;">
                <strong>${table.name}</strong>
                (${table.capacity} orang)
                ${table.location ? '- <span class="text-muted">' + table.location + '</span>' : ''}
                ${statusLabel}
                ${hasTransaction}
            </label>
        `;
    });
    
    mergeTargets.innerHTML = html;
}

// Handle merge form submission
document.getElementById('mergeForm').addEventListener('submit', function(e) {
    const sourceTableId = document.getElementById('merge_source_table_id').value;
    const targetCheckboxes = document.querySelectorAll('#mergeTargets input[type="checkbox"]:checked');
    
    if (!sourceTableId) {
        e.preventDefault();
        alert('Pilih meja sumber!');
        return false;
    }
    
    if (targetCheckboxes.length === 0) {
        e.preventDefault();
        alert('Pilih minimal satu meja yang akan digabung!');
        return false;
    }
    
    if (!confirm('Yakin ingin menggabung meja ini? Semua meja akan menggunakan transaksi yang sama.')) {
        e.preventDefault();
        return false;
    }
});
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

