<?php
/**
 * Users Management (Admin only)
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();
$auth = new Auth();
$auth->requirePermission('admin');

$pageTitle = 'Manajemen Pengguna';

$db = new Database();
$conn = $db->getConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action == 'create') {
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $name = trim($_POST['name'] ?? '');
        $role = $_POST['role'] ?? 'cashier';
        
        // Validate input
        if (empty($username) || empty($password) || empty($name)) {
            setFlash('error', 'Semua field harus diisi!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        // Validate role
        $valid_roles = ['admin', 'manager', 'cashier', 'waiters', 'dapur'];
        if (!in_array($role, $valid_roles)) {
            setFlash('error', 'Role tidak valid!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        // Check if username already exists
        $check_stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
        $check_stmt->bind_param("s", $username);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows > 0) {
            setFlash('error', 'Username sudah digunakan!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        $stmt = $conn->prepare("INSERT INTO users (username, password, name, role) VALUES (?, ?, ?, ?)");
        if (!$stmt) {
            setFlash('error', 'Error preparing statement: ' . $conn->error);
            redirect(BASE_URL . 'public/users.php');
        }
        
        $hashed_password = hashPassword($password);
        $stmt->bind_param("ssss", $username, $hashed_password, $name, $role);
        
        if ($stmt->execute()) {
            setFlash('success', 'Pengguna berhasil ditambahkan!');
        } else {
            $error_msg = 'Gagal menambahkan pengguna!';
            $db_error = $stmt->error;
            
            if (strpos($db_error, 'Duplicate') !== false || strpos($db_error, '1062') !== false) {
                $error_msg = 'Username sudah digunakan!';
            } elseif (strpos($db_error, 'role') !== false || strpos($db_error, '1265') !== false) {
                // Error 1265 is "Data truncated for column 'role'"
                $error_msg = 'Role tidak valid di database! Silakan jalankan update_database.php untuk memperbarui struktur database.';
            } else {
                $error_msg .= ' Error: ' . htmlspecialchars($db_error);
            }
            setFlash('error', $error_msg);
        }
        redirect(BASE_URL . 'public/users.php');
    }
    
    if ($action == 'update') {
        $id = intval($_POST['id'] ?? 0);
        $name = trim($_POST['name'] ?? '');
        $role = $_POST['role'] ?? 'cashier';
        $status = $_POST['status'] ?? 'active';
        
        if (empty($name)) {
            setFlash('error', 'Nama harus diisi!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        // Validate role
        $valid_roles = ['admin', 'manager', 'cashier', 'waiters', 'dapur'];
        if (!in_array($role, $valid_roles)) {
            setFlash('error', 'Role tidak valid!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        // Validate status
        $valid_status = ['active', 'inactive'];
        if (!in_array($status, $valid_status)) {
            setFlash('error', 'Status tidak valid!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        $stmt = $conn->prepare("UPDATE users SET name = ?, role = ?, status = ? WHERE id = ?");
        if (!$stmt) {
            setFlash('error', 'Error preparing statement: ' . $conn->error);
            redirect(BASE_URL . 'public/users.php');
        }
        
        $stmt->bind_param("sssi", $name, $role, $status, $id);
        
        if ($stmt->execute()) {
            setFlash('success', 'Pengguna berhasil diupdate!');
        } else {
            $error_msg = 'Gagal mengupdate pengguna!';
            $db_error = $stmt->error;
            
            if (strpos($db_error, 'role') !== false || strpos($db_error, '1265') !== false) {
                $error_msg = 'Role tidak valid di database! Silakan jalankan update_database.php untuk memperbarui struktur database.';
            } else {
                $error_msg .= ' Error: ' . htmlspecialchars($db_error);
            }
            setFlash('error', $error_msg);
        }
        redirect(BASE_URL . 'public/users.php');
    }
    
    if ($action == 'reset_password') {
        $id = intval($_POST['id'] ?? 0);
        $password = $_POST['password'] ?? '';
        
        if (empty($password)) {
            setFlash('error', 'Password harus diisi!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        if (strlen($password) < 4) {
            setFlash('error', 'Password minimal 4 karakter!');
            redirect(BASE_URL . 'public/users.php');
        }
        
        $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
        if (!$stmt) {
            setFlash('error', 'Error preparing statement: ' . $conn->error);
            redirect(BASE_URL . 'public/users.php');
        }
        
        $hashed_password = hashPassword($password);
        $stmt->bind_param("si", $hashed_password, $id);
        
        if ($stmt->execute()) {
            setFlash('success', 'Password berhasil direset!');
        } else {
            setFlash('error', 'Gagal reset password! Error: ' . $stmt->error);
        }
        redirect(BASE_URL . 'public/users.php');
    }
}

$stmt = $conn->prepare("SELECT id, username, name, role, status, last_login, created_at FROM users ORDER BY created_at DESC");
$stmt->execute();
$result = $stmt->get_result();
$users = [];
while ($row = $result->fetch_assoc()) {
    $users[] = $row;
}

include __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1>Manajemen Pengguna</h1>
    <button class="btn btn-primary" onclick="showModal('createModal')">+ Tambah Pengguna</button>
</div>

<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table">
                <thead>
                    <tr>
                        <th>Username</th>
                        <th>Nama</th>
                        <th>Role</th>
                        <th>Status</th>
                        <th>Login Terakhir</th>
                        <th>Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                        <tr>
                            <td><strong><?php echo escape($user['username']); ?></strong></td>
                            <td><?php echo escape($user['name']); ?></td>
                            <td>
                                <span class="badge badge-info"><?php echo escape($user['role']); ?></span>
                            </td>
                            <td>
                                <span class="badge <?php echo $user['status'] == 'active' ? 'badge-success' : 'badge-danger'; ?>">
                                    <?php echo escape($user['status']); ?>
                                </span>
                            </td>
                            <td><?php echo $user['last_login'] ? formatDate($user['last_login']) : '-'; ?></td>
                            <td>
                                <button class="btn btn-xs btn-primary" onclick="editUser(<?php echo htmlspecialchars(json_encode($user)); ?>)">Edit</button>
                                <button class="btn btn-xs btn-warning" onclick="resetPassword(<?php echo $user['id']; ?>)">Reset Password</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Create Modal -->
<div id="createModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>Tambah Pengguna</h2>
            <span class="close" onclick="closeModal('createModal')">&times;</span>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="create">
            <div class="modal-body">
                <div class="form-group">
                    <label>Username *</label>
                    <input type="text" name="username" required>
                </div>
                <div class="form-group">
                    <label>Password *</label>
                    <input type="password" name="password" required>
                </div>
                <div class="form-group">
                    <label>Nama *</label>
                    <input type="text" name="name" required>
                </div>
                <div class="form-group">
                    <label>Role *</label>
                    <select name="role" required>
                        <option value="cashier">Cashier</option>
                        <option value="manager">Manager</option>
                        <option value="admin">Admin</option>
                        <option value="waiters">Waiters</option>
                        <option value="dapur">Dapur</option>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('createModal')">Batal</button>
                <button type="submit" class="btn btn-primary">Simpan</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>Edit Pengguna</h2>
            <span class="close" onclick="closeModal('editModal')">&times;</span>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="update">
            <input type="hidden" name="id" id="edit_id">
            <div class="modal-body">
                <div class="form-group">
                    <label>Nama *</label>
                    <input type="text" name="name" id="edit_name" required>
                </div>
                <div class="form-group">
                    <label>Role *</label>
                    <select name="role" id="edit_role" required>
                        <option value="cashier">Cashier</option>
                        <option value="manager">Manager</option>
                        <option value="admin">Admin</option>
                        <option value="waiters">Waiters</option>
                        <option value="dapur">Dapur</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Status *</label>
                    <select name="status" id="edit_status" required>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('editModal')">Batal</button>
                <button type="submit" class="btn btn-primary">Update</button>
            </div>
        </form>
    </div>
</div>

<!-- Reset Password Modal -->
<div id="resetPasswordModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>Reset Password</h2>
            <span class="close" onclick="closeModal('resetPasswordModal')">&times;</span>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="reset_password">
            <input type="hidden" name="id" id="reset_id">
            <div class="modal-body">
                <div class="form-group">
                    <label>Password Baru *</label>
                    <input type="password" name="password" required>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('resetPasswordModal')">Batal</button>
                <button type="submit" class="btn btn-primary">Reset</button>
            </div>
        </form>
    </div>
</div>

<script>
function editUser(user) {
    document.getElementById('edit_id').value = user.id;
    document.getElementById('edit_name').value = user.name;
    document.getElementById('edit_role').value = user.role;
    document.getElementById('edit_status').value = user.status;
    showModal('editModal');
}

function resetPassword(userId) {
    document.getElementById('reset_id').value = userId;
    showModal('resetPasswordModal');
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

