<?php
/**
 * Waiter View - Table Status and Order Delivery
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/functions.php';

requireLogin();

// Allow waiters, manager, and admin
$currentUser = getCurrentUser();
$allowed_roles = ['waiters', 'manager', 'admin'];
if (!in_array($currentUser['role'], $allowed_roles)) {
    setFlash('error', 'Anda tidak memiliki akses untuk halaman ini.');
    redirect(BASE_URL . 'public/dashboard.php');
}

$pageTitle = 'Waiter - Status Meja & Pengantaran';

$table = new Table();
$transaction = new Transaction();

// Handle mark as delivered
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'mark_delivered') {
    $item_id = intval($_POST['item_id'] ?? 0);
    if ($transaction->updateItemStatus($item_id, 'completed')) {
        setFlash('success', 'Item berhasil ditandai sebagai terkirim!');
    } else {
        setFlash('error', 'Gagal mengupdate status item!');
    }
    redirect(BASE_URL . 'public/waiter.php');
}

// Handle table status update
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'update_table_status') {
    $table_id = intval($_POST['table_id'] ?? 0);
    $status = $_POST['status'] ?? 'available';
    if ($table->updateStatus($table_id, $status)) {
        setFlash('success', 'Status meja berhasil diupdate!');
    } else {
        setFlash('error', 'Gagal mengupdate status meja!');
    }
    redirect(BASE_URL . 'public/waiter.php');
}

$tables_with_ready_orders = $table->getTablesWithReadyOrders();
$all_tables = $table->getAll();
$date = date('Y-m-d');
$isWaiter = ($currentUser['role'] == 'waiters');

include __DIR__ . '/includes/header.php';
?>

<?php if ($isWaiter): ?>
<style>
/* Hide navbar for waiters */
body .navbar {
    display: none !important;
}

body .container {
    max-width: 100% !important;
    margin: 0 !important;
    padding: 1rem !important;
}

/* Prevent zoom issues */
html, body {
    zoom: 1 !important;
    -webkit-text-size-adjust: 100% !important;
    -moz-text-size-adjust: 100% !important;
    -ms-text-size-adjust: 100% !important;
    text-size-adjust: 100% !important;
}

/* Elegant background for waiter */
body {
    background: linear-gradient(135deg, #F0F4FF 0%, #E0E7FF 30%, #F8FAFF 60%, #EDE9FE 100%) !important;
    background-attachment: fixed;
    position: relative;
}

body::before {
    content: '';
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: 
        radial-gradient(circle at 20% 50%, rgba(99, 102, 241, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 80% 80%, rgba(139, 92, 246, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 40% 20%, rgba(236, 72, 153, 0.05) 0%, transparent 50%);
    pointer-events: none;
    z-index: 0;
}

body > * {
    position: relative;
    z-index: 1;
}
</style>
<?php endif; ?>

<?php if ($isWaiter): ?>
<!-- Waiter Header - Compact -->
<div class="waiter-pos-header-compact">
    <div class="waiter-pos-title-compact">
        <h1>🍽️ Waiter</h1>
        <span class="waiter-name"><?php echo escape($currentUser['name']); ?></span>
    </div>
    <div class="waiter-pos-controls-compact">
        <a href="<?php echo BASE_URL; ?>public/dashboard.php" class="btn-compact btn-dashboard" title="Kembali ke Dashboard">
            <span class="btn-icon">🏠</span>
            <span class="btn-label">Dashboard</span>
        </a>
        <a href="<?php echo BASE_URL; ?>public/pos.php" class="btn-compact btn-pos" title="POS">
            <span class="btn-icon">🛒</span>
            <span class="btn-label">POS</span>
        </a>
        <?php if (in_array($currentUser['role'], ['dapur', 'waiters', 'manager', 'admin'])): ?>
        <a href="<?php echo BASE_URL; ?>public/kitchen.php" class="btn-compact btn-kitchen" title="Dapur - Status Pesanan">
            <span class="btn-icon">🏪</span>
            <span class="btn-label">Dapur</span>
        </a>
        <?php endif; ?>
        <a href="<?php echo BASE_URL; ?>public/transactions.php" class="btn-compact btn-transactions" title="Transaksi">
            <span class="btn-icon">📋</span>
            <span class="btn-label">Transaksi</span>
        </a>
        <button class="btn-compact btn-refresh" onclick="location.reload()" title="Refresh">
            <span class="btn-icon">🔄</span>
            <span class="btn-label">Refresh</span>
        </button>
        <button class="btn-compact btn-fullscreen" id="fullscreenBtn" onclick="toggleFullscreen()" title="Fullscreen">
            <span class="btn-icon" id="fullscreenIcon">⛶</span>
            <span class="btn-label">Fullscreen</span>
        </button>
    </div>
</div>
<?php else: ?>
<div class="page-header">
    <h1>🍽️ Waiter - Status Meja & Pengantaran</h1>
    <div style="display: flex; gap: 1rem; align-items: center;">
        <span class="badge badge-info"><?php echo formatDate($date, 'd/m/Y'); ?></span>
        <button onclick="location.reload()" class="btn btn-primary">Refresh</button>
    </div>
</div>
<?php endif; ?>

<!-- Tables with Ready Orders -->
<?php if (!empty($tables_with_ready_orders)): ?>
    <div class="card <?php echo $isWaiter ? 'waiter-card-compact' : ''; ?>" style="margin-bottom: 2rem; border-left: 5px solid #28a745;">
        <div class="card-header <?php echo $isWaiter ? 'waiter-card-header-compact' : ''; ?>" style="<?php echo $isWaiter ? '' : 'background: #d4edda;'; ?>">
            <h2>⚠️ Pesanan Siap Diantar</h2>
            <span class="badge badge-success"><?php echo count($tables_with_ready_orders); ?> Meja</span>
        </div>
        <div class="card-body <?php echo $isWaiter ? 'waiter-card-body-compact' : ''; ?>">
            <div class="ready-orders-grid">
                <?php foreach ($tables_with_ready_orders as $table_data): ?>
                    <?php 
                    $order = $transaction->getById($table_data['transaction_id']);
                    $items = $transaction->getItems($table_data['transaction_id']);
                    $ready_items = array_filter($items, function($item) {
                        return $item['status'] == 'ready';
                    });
                    ?>
                    <div class="ready-order-card">
                        <div class="ready-order-header">
                            <h3>🪑 <?php echo escape($table_data['name']); ?></h3>
                            <div>
                                <span class="badge badge-success"><?php echo $table_data['ready_items']; ?> Item Siap</span>
                                <span class="badge badge-info"><?php echo formatDate($order['created_at'], 'H:i'); ?></span>
                            </div>
                        </div>
                        <div class="ready-order-body">
                            <p><strong>Invoice:</strong> <?php echo escape($order['invoice_number']); ?></p>
                            <?php 
                            $merged_tables = $transaction->getTransactionTables($table_data['transaction_id']);
                            if (count($merged_tables) > 1): 
                            ?>
                                <p><strong>🪑 Meja Gabungan:</strong>
                                    <?php foreach ($merged_tables as $merged): ?>
                                        <span class="badge badge-warning" style="margin-right: 5px;">
                                            <?php echo escape($merged['name']); ?>
                                            <?php if (!empty($merged['location'])): ?>
                                                (<?php echo escape($merged['location']); ?>)
                                            <?php endif; ?>
                                        </span>
                                    <?php endforeach; ?>
                                </p>
                            <?php else: ?>
                                <p><strong>🪑 Meja:</strong> <?php echo escape($table_data['name']); ?></p>
                                <?php if ($table_data['location']): ?>
                                    <p><strong>Lokasi:</strong> <?php echo escape($table_data['location']); ?></p>
                                <?php endif; ?>
                            <?php endif; ?>
                            <div class="ready-items-list">
                                <strong>Item Siap Diantar:</strong>
                                <ul>
                                    <?php foreach ($ready_items as $item): ?>
                                        <li>
                                            <span><?php echo escape($item['product_name']); ?> (x<?php echo $item['quantity']; ?>)</span>
                                            <form method="POST" style="display: inline; margin-left: 1rem;">
                                                <input type="hidden" name="action" value="mark_delivered">
                                                <input type="hidden" name="item_id" value="<?php echo $item['id']; ?>">
                                                <button type="submit" class="btn btn-xs btn-success">✓ Tandai Terkirim</button>
                                            </form>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- All Tables Status -->
<div class="card <?php echo $isWaiter ? 'waiter-card-compact' : ''; ?>">
    <div class="card-header <?php echo $isWaiter ? 'waiter-card-header-compact' : ''; ?>">
        <h2>Status Semua Meja</h2>
    </div>
    <div class="card-body <?php echo $isWaiter ? 'waiter-card-body-compact' : ''; ?>">
        <div class="tables-status-grid">
            <?php foreach ($all_tables as $tbl): ?>
                <?php
                $table_status = $table->getTableStatus($tbl['id']);
                $status_class = '';
                $status_label = '';
                switch ($tbl['status']) {
                    case 'available':
                        $status_class = 'status-available';
                        $status_label = 'Kosong';
                        break;
                    case 'occupied':
                        $status_class = 'status-occupied';
                        $status_label = 'Terisi';
                        break;
                    case 'reserved':
                        $status_class = 'status-reserved';
                        $status_label = 'Dipesan';
                        break;
                    case 'cleaning':
                        $status_class = 'status-cleaning';
                        $status_label = 'Bersihkan';
                        break;
                }
                ?>
                <div class="table-status-card <?php echo $status_class; ?>">
                    <div class="table-status-header">
                        <h3><?php echo escape($tbl['name']); ?></h3>
                        <span class="table-status-badge status-<?php echo $tbl['status']; ?>">
                            <?php echo $status_label; ?>
                        </span>
                    </div>
                    <div class="table-status-body">
                        <p><strong>Kapasitas:</strong> <?php echo $tbl['capacity']; ?> orang</p>
                        <?php if ($tbl['location']): ?>
                            <p><strong>Lokasi:</strong> <?php echo escape($tbl['location']); ?></p>
                        <?php endif; ?>
                        
                        <?php if ($table_status && $table_status['current_transaction_id']): 
                            $merged_tables = $transaction->getTransactionTables($table_status['current_transaction_id']);
                            $is_merged = count($merged_tables) > 1;
                        ?>
                            <div class="table-order-info">
                                <hr>
                                <p><strong>Order Aktif:</strong></p>
                                <p><small>Invoice: <?php echo escape($table_status['invoice_number']); ?></small></p>
                                <p><small>Waktu: <?php echo formatDate($table_status['order_time'], 'H:i'); ?></small></p>
                                <?php if ($is_merged): ?>
                                    <p><small><strong>🔗 Meja Gabungan:</strong> 
                                        <?php 
                                        $merged_names = array_map(function($t) { 
                                            return $t['name']; 
                                        }, $merged_tables); 
                                        echo escape(implode(', ', $merged_names)); 
                                        ?>
                                    </small></p>
                                <?php endif; ?>
                                <div class="order-status-bars">
                                    <?php if ($table_status['pending_items'] > 0): ?>
                                        <span class="status-bar status-pending">Pending: <?php echo $table_status['pending_items']; ?></span>
                                    <?php endif; ?>
                                    <?php if ($table_status['preparing_items'] > 0): ?>
                                        <span class="status-bar status-preparing">Memasak: <?php echo $table_status['preparing_items']; ?></span>
                                    <?php endif; ?>
                                    <?php if ($table_status['ready_items'] > 0): ?>
                                        <span class="status-bar status-ready">⚠️ Siap: <?php echo $table_status['ready_items']; ?></span>
                                    <?php endif; ?>
                                    <?php if ($table_status['completed_items'] > 0): ?>
                                        <span class="status-bar status-completed">Selesai: <?php echo $table_status['completed_items']; ?></span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <div class="table-status-actions">
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="update_table_status">
                                <input type="hidden" name="table_id" value="<?php echo $tbl['id']; ?>">
                                <select name="status" class="form-control form-control-sm" onchange="this.form.submit()">
                                    <option value="available" <?php echo $tbl['status'] == 'available' ? 'selected' : ''; ?>>Kosong</option>
                                    <option value="occupied" <?php echo $tbl['status'] == 'occupied' ? 'selected' : ''; ?>>Terisi</option>
                                    <option value="reserved" <?php echo $tbl['status'] == 'reserved' ? 'selected' : ''; ?>>Dipesan</option>
                                    <option value="cleaning" <?php echo $tbl['status'] == 'cleaning' ? 'selected' : ''; ?>>Bersihkan</option>
                                </select>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<?php if ($isWaiter): ?>
<style>
/* ============================================
   COMPACT & TABLET-FRIENDLY WAITER STYLES
   ============================================ */

/* Compact Header */
.waiter-pos-header-compact {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 30%, #7C3AED 60%, #EC4899 100%);
    padding: 0.875rem 1.25rem;
    border-radius: 16px;
    margin-bottom: 1rem;
    box-shadow: 0 6px 24px rgba(99, 102, 241, 0.35), 0 0 0 1px rgba(255, 255, 255, 0.1) inset;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: sticky;
    top: 0;
    z-index: 100;
    backdrop-filter: blur(10px);
    position: relative;
    overflow: hidden;
}

.waiter-pos-header-compact::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    animation: headerShine 3s infinite;
}

@keyframes headerShine {
    0% {
        left: -100%;
    }
    100% {
        left: 100%;
    }
}

.waiter-pos-title-compact {
    position: relative;
    z-index: 1;
}

.waiter-pos-title-compact h1 {
    color: white;
    font-size: 1.5rem;
    margin: 0;
    font-weight: 800;
    display: inline-block;
    text-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
    letter-spacing: -0.5px;
}

.waiter-name {
    color: rgba(255,255,255,0.9);
    font-size: 0.875rem;
    margin-left: 0.75rem;
    font-weight: 500;
}

.waiter-pos-controls-compact {
    display: flex;
    gap: 0.5rem;
}

.btn-compact {
    background: rgba(255,255,255,0.2);
    color: white;
    border: 2px solid rgba(255,255,255,0.5);
    border-radius: 8px;
    padding: 0.5rem 0.625rem;
    font-size: 1.25rem;
    cursor: pointer;
    transition: all 0.2s;
    min-width: 60px;
    min-height: 60px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 0.25rem;
    text-decoration: none;
}

.btn-compact:hover {
    background: white;
    color: #6366F1;
    transform: scale(1.05);
    text-decoration: none;
}

.btn-icon {
    font-size: 1.25rem;
    line-height: 1;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-label {
    font-size: 0.625rem;
    font-weight: 600;
    line-height: 1;
    text-align: center;
    white-space: nowrap;
    letter-spacing: 0.02em;
}

/* Compact Cards */
.waiter-card-compact {
    background: linear-gradient(135deg, #FFFFFF 0%, #F8FAFF 100%);
    border-radius: 16px;
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.1), 0 0 0 1px rgba(224, 231, 255, 0.5) inset;
    overflow: hidden;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid rgba(224, 231, 255, 0.8);
    position: relative;
}

.waiter-card-compact::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #6366F1, #8B5CF6, #EC4899);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.waiter-card-compact:hover {
    box-shadow: 0 8px 32px rgba(99, 102, 241, 0.2), 0 0 0 1px rgba(99, 102, 241, 0.3) inset;
    transform: translateY(-2px);
}

.waiter-card-compact:hover::after {
    opacity: 1;
}

.waiter-card-header-compact {
    background: linear-gradient(135deg, #6366F1 0%, #4F46E5 50%, #7C3AED 100%);
    color: white;
    padding: 0.875rem 1rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    overflow: hidden;
}

.waiter-card-header-compact::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: shimmer 3s infinite;
}

@keyframes shimmer {
    0%, 100% {
        transform: translate(0, 0) rotate(0deg);
    }
    50% {
        transform: translate(50%, 50%) rotate(180deg);
    }
}

.waiter-card-header-compact h2 {
    margin: 0;
    font-size: 1rem;
    font-weight: 600;
}

.waiter-card-body-compact {
    padding: 1rem;
}
</style>
<?php endif; ?>

<style>
.ready-orders-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 1.5rem;
}

.ready-order-card {
    background: #fffbf0;
    border: 3px solid #ffc107;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 4px 12px rgba(255, 193, 7, 0.3);
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { box-shadow: 0 4px 12px rgba(255, 193, 7, 0.3); }
    50% { box-shadow: 0 4px 20px rgba(255, 193, 7, 0.6); }
}

.ready-order-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 2px solid #ffc107;
}

.ready-order-header h3 {
    margin: 0;
    font-size: 1.25rem;
    color: #333;
}

.ready-items-list {
    margin-top: 1rem;
    padding: 1rem;
    background: white;
    border-radius: 8px;
}

.ready-items-list ul {
    margin: 0.5rem 0 0 0;
    padding-left: 1.5rem;
}

.ready-items-list li {
    margin: 0.5rem 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.tables-status-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 1.5rem;
}

.table-status-card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    overflow: hidden;
    border: 3px solid #e0e0e0;
    transition: transform 0.3s, box-shadow 0.3s;
}

.table-status-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 4px 16px rgba(0,0,0,0.15);
}

.table-status-card.status-available {
    border-color: #28a745;
    background: #f0fff4;
}

.table-status-card.status-occupied {
    border-color: #dc3545;
    background: #fff5f5;
}

.table-status-card.status-reserved {
    border-color: #ffc107;
    background: #fffbf0;
}

.table-status-card.status-cleaning {
    border-color: #17a2b8;
    background: #f0f9ff;
}

.table-status-header {
    padding: 1rem 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 2px solid #e0e0e0;
}

.table-status-header h3 {
    margin: 0;
    font-size: 1.125rem;
    color: #333;
}

.table-status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: bold;
    text-transform: uppercase;
}

.table-status-badge.status-available {
    background: #28a745;
    color: white;
}

.table-status-badge.status-occupied {
    background: #dc3545;
    color: white;
}

.table-status-badge.status-reserved {
    background: #ffc107;
    color: #333;
}

.table-status-badge.status-cleaning {
    background: #17a2b8;
    color: white;
}

.table-status-body {
    padding: 1.5rem;
}

.table-status-body p {
    margin: 0.5rem 0;
    font-size: 0.9rem;
}

.table-order-info {
    margin-top: 1rem;
    padding-top: 1rem;
}

.order-status-bars {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
    margin-top: 0.5rem;
}

.status-bar {
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-size: 0.75rem;
    font-weight: bold;
}

.status-bar.status-pending {
    background: #fff3cd;
    color: #856404;
}

.status-bar.status-preparing {
    background: #d1ecf1;
    color: #0c5460;
}

.status-bar.status-ready {
    background: #d4edda;
    color: #155724;
    animation: blink 1s infinite;
}

.status-bar.status-completed {
    background: #d4edda;
    color: #155724;
}

@keyframes blink {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.6; }
}

.table-status-actions {
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 1px solid #e0e0e0;
}

@media (max-width: 768px) {
    .ready-orders-grid,
    .tables-status-grid {
        grid-template-columns: 1fr;
    }
}

/* Fullscreen Mode */
body.waiter-fullscreen-active .waiter-pos-header-compact {
    margin-bottom: 0.75rem;
    padding: 0.625rem 0.875rem;
}

body.waiter-fullscreen-active .navbar {
    display: none;
}

body.waiter-fullscreen-active .container {
    max-width: 100%;
    margin: 0;
    padding: 0.5rem;
}
</style>

<script>
// Auto refresh every 30 seconds
setInterval(function() {
    if (!document.hidden) {
        location.reload();
    }
}, 30000);

// Enter fullscreen function
function enterFullscreen() {
    const elem = document.documentElement;
    
    if (!document.fullscreenElement && !document.webkitFullscreenElement && 
        !document.mozFullScreenElement && !document.msFullscreenElement) {
        // Try to enter fullscreen
        const promise = elem.requestFullscreen ? elem.requestFullscreen() :
                       elem.webkitRequestFullscreen ? elem.webkitRequestFullscreen() :
                       elem.mozRequestFullScreen ? elem.mozRequestFullScreen() :
                       elem.msRequestFullscreen ? elem.msRequestFullscreen() : null;
        
        if (promise) {
            promise.then(() => {
                // Fullscreen successful, update button state
                updateFullscreenButton();
            }).catch(err => {
                // Fullscreen failed (may require user interaction in some browsers)
                console.log('Fullscreen request failed:', err);
            });
        }
    }
}

// Exit fullscreen function
function exitFullscreen() {
    if (document.exitFullscreen) {
        document.exitFullscreen();
    } else if (document.webkitExitFullscreen) {
        document.webkitExitFullscreen();
    } else if (document.mozCancelFullScreen) {
        document.mozCancelFullScreen();
    } else if (document.msExitFullscreen) {
        document.msExitFullscreen();
    }
}

// Fullscreen functionality
function toggleFullscreen() {
    const elem = document.documentElement;
    const fullscreenBtn = document.getElementById('fullscreenBtn');
    const fullscreenIcon = document.getElementById('fullscreenIcon');
    const fullscreenText = document.getElementById('fullscreenText');
    
    if (!document.fullscreenElement && !document.webkitFullscreenElement && 
        !document.mozFullScreenElement && !document.msFullscreenElement) {
        // Enter fullscreen
        enterFullscreen();
        
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Exit Fullscreen';
        document.body.classList.add('waiter-fullscreen-active');
    } else {
        // Exit fullscreen
        exitFullscreen();
        
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Fullscreen';
        document.body.classList.remove('waiter-fullscreen-active');
    }
}

// Listen for fullscreen changes
document.addEventListener('fullscreenchange', updateFullscreenButton);
document.addEventListener('webkitfullscreenchange', updateFullscreenButton);
document.addEventListener('mozfullscreenchange', updateFullscreenButton);
document.addEventListener('MSFullscreenChange', updateFullscreenButton);

function updateFullscreenButton() {
    const isFullscreen = !!(document.fullscreenElement || document.webkitFullscreenElement || 
                            document.mozFullScreenElement || document.msFullscreenElement);
    const fullscreenIcon = document.getElementById('fullscreenIcon');
    const fullscreenText = document.getElementById('fullscreenText');
    
    if (isFullscreen) {
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Exit Fullscreen';
        document.body.classList.add('waiter-fullscreen-active');
    } else {
        if (fullscreenIcon) fullscreenIcon.textContent = '⛶';
        if (fullscreenText) fullscreenText.textContent = 'Fullscreen';
        document.body.classList.remove('waiter-fullscreen-active');
    }
}

// Keyboard shortcut for fullscreen (F11) - only for waiters
<?php if ($isWaiter): ?>
document.addEventListener('keydown', function(e) {
    if (e.key === 'F11') {
        e.preventDefault();
        toggleFullscreen();
    }
});
<?php endif; ?>
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

